/*
 * Decompiled with CFR 0.152.
 */
package org.simantics.acorn.backup;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.concurrent.Future;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.simantics.acorn.AcornSessionManagerImpl;
import org.simantics.acorn.GraphClientImpl2;
import org.simantics.acorn.exception.IllegalAcornStateException;
import org.simantics.backup.BackupException;
import org.simantics.backup.IBackupProvider;
import org.simantics.db.server.ProCoreException;
import org.simantics.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AcornBackupProvider
implements IBackupProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(AcornBackupProvider.class);
    private static final String IDENTIFIER = "AcornBackupProvider";
    private long trId = -1L;
    private final Semaphore lock = new Semaphore(1);
    private final GraphClientImpl2 client = AcornSessionManagerImpl.getInstance().getClient();

    public static Path getAcornMetadataFile(Path dbFolder) {
        return dbFolder.getParent().resolve(IDENTIFIER);
    }

    public void lock() throws BackupException {
        try {
            if (this.trId != -1L) {
                throw new IllegalStateException(this + " backup provider is already locked");
            }
            this.trId = this.client.askWriteTransaction(-1L).getTransactionId();
        }
        catch (ProCoreException e) {
            LOGGER.error("Failed to lock backup provider", (Throwable)e);
        }
    }

    public Future<BackupException> backup(Path targetPath, int revision) throws BackupException {
        boolean releaseLock = true;
        try {
            this.lock.acquire();
            Future<BackupException> r = this.client.getBackupRunnable(this.lock, targetPath, revision);
            releaseLock = false;
            Future<BackupException> future = r;
            return future;
        }
        catch (InterruptedException e) {
            releaseLock = false;
            throw new BackupException("Failed to lock Acorn for backup.", (Throwable)e);
        }
        catch (NumberFormatException e) {
            throw new BackupException("Failed to read Acorn head state file.", (Throwable)e);
        }
        catch (IOException | IllegalAcornStateException e) {
            throw new BackupException("I/O problem during Acorn backup.", (Throwable)e);
        }
        finally {
            if (releaseLock) {
                this.lock.release();
            }
        }
    }

    public void unlock() throws BackupException {
        try {
            if (this.trId == -1L) {
                throw new BackupException(this + " backup provider is not locked");
            }
            this.client.endTransaction(this.trId);
            this.trId = -1L;
        }
        catch (ProCoreException e) {
            throw new BackupException((Throwable)e);
        }
    }

    public void restore(Path fromPath, int revision) {
        try {
            Path dbRoot;
            Path restorePath = dbRoot = this.client.getDbFolder();
            if (!Files.exists(dbRoot, LinkOption.NOFOLLOW_LINKS)) {
                Files.createDirectories(dbRoot, new FileAttribute[0]);
            } else {
                Path dbRootParent = dbRoot.getParent();
                restorePath = dbRootParent == null ? Files.createTempDirectory("restore", new FileAttribute[0]) : Files.createTempDirectory(dbRootParent, "restore", new FileAttribute[0]);
            }
            Files.walkFileTree(fromPath, new RestoreCopyVisitor(restorePath, revision));
            if (dbRoot != restorePath) {
                FileUtils.deleteAll((File)dbRoot.toFile());
                Files.move(restorePath, dbRoot, new CopyOption[0]);
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to restore database revision {} from backup {}", new Object[]{revision, fromPath.toString(), e});
        }
    }

    public static class AcornBackupRunnable
    implements Runnable,
    Future<BackupException> {
        private final Semaphore lock;
        private final Path targetPath;
        private final int revision;
        private final Path baseDir;
        private final int latestFolder;
        private final int newestFolder;
        private boolean done = false;
        private final Semaphore completion = new Semaphore(0);
        private BackupException exception = null;

        public AcornBackupRunnable(Semaphore lock, Path targetPath, int revision, Path baseDir, int latestFolder, int newestFolder) {
            this.lock = lock;
            this.targetPath = targetPath;
            this.revision = revision;
            this.baseDir = baseDir;
            this.latestFolder = latestFolder;
            this.newestFolder = newestFolder;
        }

        @Override
        public void run() {
            try {
                try {
                    this.doBackup();
                    this.writeHeadstateFile();
                }
                catch (IOException e) {
                    this.exception = new BackupException("Acorn backup failed", (Throwable)e);
                    this.rollback();
                    this.done = true;
                    this.lock.release();
                    this.completion.release();
                }
            }
            finally {
                this.done = true;
                this.lock.release();
                this.completion.release();
            }
        }

        private void doBackup() throws IOException {
            Path target = this.targetPath.resolve(String.valueOf(this.revision)).resolve(AcornBackupProvider.IDENTIFIER);
            Files.createDirectories(target, new FileAttribute[0]);
            Files.walkFileTree(this.baseDir, new BackupCopyVisitor(this.baseDir, target));
        }

        private void writeHeadstateFile() throws IOException {
            Path AcornMetadataFile = AcornBackupProvider.getAcornMetadataFile(this.baseDir);
            if (!Files.exists(AcornMetadataFile, new LinkOption[0])) {
                Files.createFile(AcornMetadataFile, new FileAttribute[0]);
            }
            Files.write(AcornMetadataFile, Arrays.asList(Integer.toString(this.newestFolder)), StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.CREATE);
        }

        private void rollback() {
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            return false;
        }

        @Override
        public boolean isCancelled() {
            return false;
        }

        @Override
        public boolean isDone() {
            return this.done;
        }

        @Override
        public BackupException get() throws InterruptedException {
            this.completion.acquire();
            this.completion.release();
            return this.exception;
        }

        @Override
        public BackupException get(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException {
            if (!this.completion.tryAcquire(timeout, unit)) {
                throw new TimeoutException("Acorn backup completion waiting timed out.");
            }
            this.completion.release();
            return this.exception;
        }

        private class BackupCopyVisitor
        extends SimpleFileVisitor<Path> {
            private Path fromPath;
            private Path toPath;

            public BackupCopyVisitor(Path fromPath, Path toPath) {
                this.fromPath = fromPath;
                this.toPath = toPath;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (dir.equals(this.fromPath)) {
                    Path targetPath = this.toPath.resolve(this.fromPath.relativize(dir));
                    Files.createDirectories(targetPath, new FileAttribute[0]);
                    return FileVisitResult.CONTINUE;
                }
                try {
                    int dirNameInt = Integer.parseInt(dir.getFileName().toString());
                    if (AcornBackupRunnable.this.latestFolder < dirNameInt && dirNameInt <= AcornBackupRunnable.this.newestFolder) {
                        Path targetPath = this.toPath.resolve(this.fromPath.relativize(dir));
                        Files.createDirectories(targetPath, new FileAttribute[0]);
                        return FileVisitResult.CONTINUE;
                    }
                }
                catch (NumberFormatException numberFormatException) {}
                return FileVisitResult.SKIP_SUBTREE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Backup " + file + " to " + this.toPath.resolve(this.fromPath.relativize(file)));
                }
                Files.copy(file, this.toPath.resolve(this.fromPath.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }
        }
    }

    private class RestoreCopyVisitor
    extends SimpleFileVisitor<Path> {
        private final Path toPath;
        private final int revision;
        private Path currentSubFolder;

        public RestoreCopyVisitor(Path toPath, int revision) {
            this.toPath = toPath;
            this.revision = revision;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            Path dirName = dir.getFileName();
            if (dirName.toString().equals(AcornBackupProvider.IDENTIFIER)) {
                this.currentSubFolder = dir;
                return FileVisitResult.CONTINUE;
            }
            if (dir.getParent().getFileName().toString().equals(AcornBackupProvider.IDENTIFIER)) {
                Path targetPath = this.toPath.resolve(dirName);
                Files.createDirectories(targetPath, new FileAttribute[0]);
                return FileVisitResult.CONTINUE;
            }
            if (dirName.toString().length() == 1 && Character.isDigit(dirName.toString().charAt(0))) {
                int dirNameInt = Integer.parseInt(dirName.toString());
                if (dirNameInt <= this.revision) {
                    return FileVisitResult.CONTINUE;
                }
                return FileVisitResult.SKIP_SUBTREE;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (file.getFileName().toString().endsWith(".tar.gz")) {
                return FileVisitResult.CONTINUE;
            }
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Restore " + file + " to " + this.toPath.resolve(this.currentSubFolder.relativize(file)));
            }
            Files.copy(file, this.toPath.resolve(this.currentSubFolder.relativize(file)), StandardCopyOption.REPLACE_EXISTING);
            return FileVisitResult.CONTINUE;
        }
    }
}

