/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.selectionview;

import java.util.function.Consumer;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.common.views.IFilterArea;
import org.simantics.browsing.ui.common.views.IFilterAreaProvider;
import org.simantics.browsing.ui.swt.PartNameListener;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupportImpl;
import org.simantics.db.Disposable;
import org.simantics.db.common.procedure.adapter.ListenerSupport;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.request.Read;

/**
 * Important points of customization:
 * <ul>
 * <li>{@link #createControls(Composite, IWorkbenchSite, ISessionContext, WidgetSupport)}</li>
 * <li>{@link #getSelectionProvider()}</li>
 * <li>{@link #getPartNameReadRequest(ISelection)}</li>
 * </ul>

 * @author Antti Villberg
 * @author Tuukka Lehtonen
 * @see ModelledTabContributor ModelledTabContributor for graph-modelled tab contributions
 */
public abstract class PropertyTabContributorImpl implements PropertyTabContributor, Disposable /*, ListenerSupport*/ {

    // For ListenerSupport (supporting DB request listeners)
//    protected boolean    disposed = false;
//    protected ISelection input    = StructuredSelection.EMPTY;
//    private Object identity;

    abstract public void createControls(Composite body, IWorkbenchSite site, final ISessionContext context, WidgetSupport support);

    public PropertyTabContributorImpl() {
    }

    public IFilterArea getFilterArea() {
        return null;
    }

    public void requestFocus() {
    }

    public ISelectionProvider getSelectionProvider() {
        return null;
    }

    public String getPartName(ISelection forSelection) {
    	return null;
    }
    
    public Read<String> getPartNameReadRequest(ISelection forSelection) {
        return null;
    }

//    public void updatePartName(ISelection forSelection, Callback<String> updateCallback) {
//        Read<String> read = getPartNameReadRequest(forSelection);
//        if (read == null) {
//            updateCallback.run("Override to control part name (PropertyTabContributorImpl.updatePartName)");
//        } else {
//            Simantics.getSession().asyncRequest(read, new PartNameListener(updateCallback, this));
//        }
//    }

//    public void updatePartName(Callback<String> updateCallback) {
//        updatePartName(input, updateCallback);
//    }

//    protected void dispose() {
//        this.disposed = true;
//    }

//    @Override
//    public void exception(Throwable t) {
//        ErrorLogger.defaultLogError(getClass().getSimpleName() + " received unexpected exception.", t);
//    }

//    @Override
//    public boolean isDisposed() {
//        return disposed;
//    }

    public void createControl(Composite parent, final IWorkbenchSite site, final ISessionContext context, final WidgetSupportImpl support) {

        class TabComposite extends Composite {
            public TabComposite(Composite parent) {
                super(parent, 0);

                GridLayoutFactory.fillDefaults().applyTo(parent);
                GridDataFactory.fillDefaults().span(1, 1).grab(true, true).applyTo(this);

                Composite body = this;
                GridLayoutFactory.fillDefaults().spacing(0, 0).equalWidth(false).numColumns(1).applyTo(body);

                try {
                    PropertyTabContributorImpl.this.createControls(body, site, context, support);
                } catch (Throwable t) {
                    ErrorLogger.defaultLogError(t);
                }
            }
        }

        final TabComposite tc = new TabComposite(parent);
        tc.addListener(SWT.Dispose, new Listener() {
        	public void handleEvent(Event e) {
        		if(support instanceof Disposable)
        			((Disposable)support).dispose();
        		PropertyTabContributorImpl.this.dispose();
            }
        });
        
    }

    @Override
    public IPropertyTab create(Composite parent, IWorkbenchSite site, ISessionContext context, Object input) {
        IPropertyTab tab = new Tab(site, parent);
        tab.createControl((Composite) tab.getControl(), context);
        return tab;
    }

    protected WidgetSupportImpl createSupport() {
    	return new WidgetSupportImpl();
    }
    
    class Tab extends Composite implements IPropertyTab2, IFilterAreaProvider, ListenerSupport {

        final IWorkbenchSite    site;
        final WidgetSupportImpl support = createSupport();
        protected ISelection input    = StructuredSelection.EMPTY;

        public Tab(IWorkbenchSite site, Composite parent) {
            super(parent, SWT.NONE);
            this.site = site;
        }

        @Override
        public void createControl(Composite parent, ISessionContext context) {
            PropertyTabContributorImpl.this.createControl(parent, site, context, support);
        }

        @Override
        public Control getControl() {
            return this;
        }

        @Override
        public boolean isDisposed() {
            return super.isDisposed();
        }

        @Override
        public void requestFocus() {
            PropertyTabContributorImpl.this.requestFocus();
        }

        @Override
        public void setInput(ISessionContext context, ISelection selection, boolean force) {
            this.input = selection;
            support.fireInput(context, selection);
        }

        @Override
        public ISelectionProvider getSelectionProvider() {
            return PropertyTabContributorImpl.this.getSelectionProvider();
        }

        @Override
        public IFilterArea getFilterArea() {
            return PropertyTabContributorImpl.this.getFilterArea();
        }

        @Override
        public void updatePartName(Consumer<String> updateCallback) {
        	String partName = PropertyTabContributorImpl.this.getPartName(input);
        	if(partName != null) {
        		updateCallback.accept(partName);
        		return;
        	}
        	Read<String> read = getPartNameReadRequest(input);
        	if (read == null) {
        		updateCallback.accept("Override to control part name (PropertyTabContributorImpl.updatePartName)");
        	} else {
        		Simantics.getSession().asyncRequest(read, new PartNameListener(updateCallback, this));
        	}
            //PropertyTabContributorImpl.this.updatePartName(input, updateCallback);
        }

		@Override
		public void exception(Throwable t) {
	        ErrorLogger.defaultLogError(getClass().getSimpleName() + " received unexpected exception.", t);
		}

    }
    
    @Override
    public void dispose() {
    }
    
}
