package org.simantics.selectionview;

import java.io.IOException;

import org.simantics.common.format.Formatter;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.NumberBinding;
import org.simantics.databoard.binding.StringBinding;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.MutableStringBinding;
import org.simantics.databoard.parser.DataValuePrinter;
import org.simantics.databoard.parser.repository.DataTypeSyntaxError;
import org.simantics.databoard.parser.repository.DataValueRepository;
import org.simantics.databoard.primitives.MutableString;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.util.ObjectUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.adaption.SimpleContextualAdapter;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.variable.ModelledVariablePropertyDescriptor;
import org.simantics.db.layer0.variable.StandardGraphPropertyVariable;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.scl.runtime.function.Function1;

public class DisplayValueVariableAdapter extends SimpleContextualAdapter<Variable, ModelledVariablePropertyDescriptor> {
	
	final String key;
	
	public DisplayValueVariableAdapter() {
		key = SelectionViewResources.URIs.HasDisplayValue;
	}

	public DisplayValueVariableAdapter(String key) {
		this.key = key;
	}

	class Impl extends StandardGraphPropertyVariable {

		private Datatype datatype;

		public Impl(ReadGraph graph, Variable parent, Resource parentResource, Datatype datatype) throws DatabaseException {
			super(graph, parent, null, parentResource, graph.getResource(key));
			this.datatype = datatype;
		}
		
		@SuppressWarnings("unchecked")
		@Override
		public <T> T getValue(ReadGraph graph) throws DatabaseException {
			return (T)standardGetDisplayValue1(graph, this);
		}
		
		@SuppressWarnings("unchecked")
		@Override
		public <T> T getValue(ReadGraph graph, Binding binding) throws DatabaseException {
			return (T)standardGetDisplayValue2(graph, this, binding);
		}
		
		@Override
		public Datatype getDatatype(ReadGraph graph) throws DatabaseException {
			return datatype;
		}
			
   		@Override
   		public void setValue(WriteGraph graph, Object value) throws DatabaseException {
			standardSetDisplayValue2(graph, this, value);
   		}
   		

		@Override
		public void setValue(WriteGraph graph, Object _value, Binding _binding) throws DatabaseException {
			standardSetDisplayValue3(graph, this, _value, _binding);
		}
		
	}
	
	public static boolean isPrimitive(Datatype dt) {
		if(Datatypes.STRING.equals(dt)) return true;
		else return false;
	}

	private static String possibleExpression(ReadGraph graph, Variable variable) throws DatabaseException {
		
		Layer0 L0 = Layer0.getInstance(graph);
		ModelingResources MOD = ModelingResources.getInstance(graph);
		Resource object = variable.getPossibleRepresents(graph);
		if(object != null && graph.isInstanceOf(object, MOD.SCLValue)) {
			String expression = graph.getPossibleRelatedValue(object, L0.SCLValue_expression);
			if (expression != null)
				return "=" + expression;
		}
		return null;
		
	}

	@Override
	public Variable adapt(ReadGraph graph, Resource source, ModelledVariablePropertyDescriptor context) throws DatabaseException {
		
//    	String value = getDisplayValue(graph, context.getVariable());
    	return new Impl(graph, context.getVariable(), context.getSubject(), Datatypes.STRING);
		
	}

    public static Object standardGetDisplayValue1(ReadGraph graph, Variable property_) throws DatabaseException {

    	Variable property = property_.getParent(graph); 
    	
    	String expression = possibleExpression(graph, property);
    	if(expression != null) return expression;
    	
    	Object value = null;

    	Resource formatter = property.getPossiblePropertyValue(graph, Variables.FORMATTER);
    	if(formatter != null) {
    		Formatter fmt = graph.adaptContextual(formatter, property, Variable.class, Formatter.class);
    		value = fmt.format(property.getValue(graph));
    	}
    	if(value == null) {
    		SelectionViewResources SEL = SelectionViewResources.getInstance(graph);
    		Function1<Object,String> formatterFunction = property.getPossiblePropertyValue(graph, SEL.formatter);
    		if(formatterFunction != null) {
    			value = formatterFunction.apply(property.getValue(graph)); 
    		}
    	}
    	if(value == null) {

    		Datatype dt = property.getPossibleDatatype(graph);
    		if(dt != null) {
    			if(!isPrimitive(dt)) {
    				Binding binding = Bindings.getBinding(dt);
    				try {
    					value = DataValuePrinter.writeValueSingleLine(binding, property.getValue(graph));
    				} catch (IOException e) {
    					e.printStackTrace();
    				} catch (BindingException e) {
    					e.printStackTrace();
    				}
    			}
    		}

    	}

    	return value != null ? value.toString() : "null";
    	
    }

    public static Object standardGetDisplayValue2(ReadGraph graph, Variable property, Binding binding) throws DatabaseException {
    	
		try {
			return Bindings.adapt(standardGetDisplayValue1(graph, property), Bindings.STRING, binding);
		} catch (AdaptException e) {
			throw new DatabaseException(e);
		}
    	
    }

    public static void standardSetDisplayValue2(WriteGraph graph, Variable property_, Object _value) throws DatabaseException {
			
		try {
			Binding binding = Bindings.getBinding(_value.getClass());
			standardSetDisplayValue3(graph, property_, _value, binding);
		} catch (BindingConstructionException e) {
			throw new DatabaseException(e);
		}

    }

    public static void standardSetDisplayValue3(WriteGraph graph, Variable property_, Object _value, Binding binding_) throws DatabaseException {

        try {
        	
        	Variable parent = property_.getParent(graph);
        	
   			if(!(_value instanceof String)) throw new DatabaseException("setValue for HasDisplayValue only accepts String (got " + _value.getClass().getSimpleName() + ")");

   			String text = (String)_value;
   			if(text.startsWith("=")) {
   			    Layer0Utils.setExpression(graph, parent, text, ModelingResources.getInstance(graph).SCLValue);
   				return;
   			}

        	String parsedLabel = (String)_value;
            Object value = parsedLabel;
        	
            Datatype type = parent.getPossibleDatatype(graph);
            if (type != null) {

	            Binding binding = Bindings.getBinding(type);

	            if (binding instanceof StringBinding) {
	            	
	                if (binding instanceof MutableStringBinding)
	                    value = new MutableString(parsedLabel);
	                else
	                    value = parsedLabel;
	                
	            } else {
	            	
	                if (binding instanceof NumberBinding) {
	                    parsedLabel = parsedLabel.replace(",", ".");
	                }

	                value = binding.parseValue(parsedLabel, new DataValueRepository());
	            }

	            //System.out.println("VariableWrite " + ObjectUtils.toString(value));
	            parent.setValue(graph, value, binding);
	            
            } else {

            	parent.setValue(graph, value);
            	
            }


            // Add a comment to metadata.
            CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
            graph.addMetadata(cm.add("Set value " + ObjectUtils.toString(value)));
            
        } catch (DataTypeSyntaxError e) {
            throw new DatabaseException(e);
        } catch (BindingException e) {
            throw new DatabaseException(e);
        }
    	
    }

    public static Datatype standardGetDisplayValueDatatype(ReadGraph graph, Variable property_) throws DatabaseException {
    	return Datatypes.STRING;
    }
	
}
