/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;

import org.junit.Test;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.IntegerBinding;
import org.simantics.databoard.binding.MapBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.util.RandomValue;
import org.simantics.databoard.parser.repository.DataTypeSyntaxError;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.serialization.SerializerConstructionException;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.StringType;

/**
 * This Test Case tests
 *  o ReflectionBinding, GeneralBinding
 *  o Boolean, Byte, Int, Long, Float, Double, String, Variant, Array, Map, Union, Record
 *  o Create, Read
 *  o Equals
 *  o HashCode
 *  o CompareTo (Binding & DataValueUtil)
 *  o Clone
 *  o Create Default
 *  o Adaption between bindings
 *  o Validator
 *  o Serializer
 *  o NullBinding Validator
 *  o Ascii Print/Parse
 *  o Recursion
 *  
 * TODO Test Range & Length Annotations
 * TODO Test that map Sorts values
 * TODO Test Data Types
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class TestBinding extends TestData {
	
	public static boolean OMIT_PARSER_TEST = true;
	
	// boolean
	@Test public void test_boolean() throws Exception {
		assert_same(boolean_binding_ref, boolean_value_ref_0, boolean_binding_gen, boolean_value_gen_0);
		assert_same(boolean_binding_ref, boolean_value_ref_1, boolean_binding_gen, boolean_value_gen_1);
		assert_same(boolean_binding_ref, boolean_value_ref_0, boolean_binding_ref, boolean_value_ref_0);
		assert_same(boolean_binding_gen, boolean_value_gen_0, boolean_binding_gen, boolean_value_gen_0);
		assert_not_same(boolean_binding_ref, boolean_value_ref_0, boolean_binding_gen, boolean_value_gen_1);
		assert_not_same(boolean_binding_ref, boolean_value_ref_1, boolean_binding_gen, boolean_value_gen_0);		
		assert_not_same(boolean_binding_ref, boolean_value_ref_0, boolean_binding_ref, boolean_value_ref_1);
		assert_not_same(boolean_binding_gen, boolean_value_gen_0, boolean_binding_gen, boolean_value_gen_1);
		assert_valid(boolean_binding_ref, boolean_value_ref_0);
		assert_valid(boolean_binding_ref, boolean_value_ref_1);
		assert_valid(boolean_binding_gen, boolean_value_gen_0);
		assert_valid(boolean_binding_gen, boolean_value_gen_1);
	}
		
	// byte
	@Test public void test_byte() throws Exception {
		assert_same(byte_binding_ref, byte_value_ref_0, byte_binding_gen, byte_value_gen_0);
		assert_same(byte_binding_ref, byte_value_ref_1, byte_binding_gen, byte_value_gen_1);
		assert_same(byte_binding_ref, byte_value_ref_0, byte_binding_ref, byte_value_ref_0);
		assert_same(byte_binding_gen, byte_value_gen_0, byte_binding_gen, byte_value_gen_0);
		assert_not_same(byte_binding_ref, byte_value_ref_0, byte_binding_gen, byte_value_gen_1);
		assert_not_same(byte_binding_ref, byte_value_ref_1, byte_binding_gen, byte_value_gen_0);		
		assert_not_same(byte_binding_ref, byte_value_ref_0, byte_binding_ref, byte_value_ref_1);
		assert_not_same(byte_binding_gen, byte_value_gen_0, byte_binding_gen, byte_value_gen_1);
		assert_valid(byte_binding_ref, byte_value_ref_0);
		assert_valid(byte_binding_ref, byte_value_ref_1);
		assert_valid(byte_binding_gen, byte_value_gen_0);
		assert_valid(byte_binding_gen, byte_value_gen_1);
	}
	
	// integer
	@Test public void test_integer() throws Exception {
		assert_same(integer_binding_ref, integer_value_ref_0, integer_binding_gen, integer_value_gen_0);
		assert_same(integer_binding_ref, integer_value_ref_1, integer_binding_gen, integer_value_gen_1);
		assert_same(integer_binding_ref, integer_value_ref_0, integer_binding_ref, integer_value_ref_0);
		assert_same(integer_binding_gen, integer_value_gen_0, integer_binding_gen, integer_value_gen_0);
		assert_not_same(integer_binding_ref, integer_value_ref_0, integer_binding_gen, integer_value_gen_1);
		assert_not_same(integer_binding_ref, integer_value_ref_1, integer_binding_gen, integer_value_gen_0);		
		assert_not_same(integer_binding_ref, integer_value_ref_0, integer_binding_ref, integer_value_ref_1);
		assert_not_same(integer_binding_gen, integer_value_gen_0, integer_binding_gen, integer_value_gen_1);
		assert_valid(integer_binding_ref, integer_value_ref_0);
		assert_valid(integer_binding_ref, integer_value_ref_1);
		assert_valid(integer_binding_gen, integer_value_gen_0);
		assert_valid(integer_binding_gen, integer_value_gen_1);
	}
	
	// long
	@Test public void test_long() throws Exception {
		assert_same(long_binding_ref, long_value_ref_0, long_binding_gen, long_value_gen_0);
		assert_same(long_binding_ref, long_value_ref_1, long_binding_gen, long_value_gen_1);
		assert_same(long_binding_ref, long_value_ref_0, long_binding_ref, long_value_ref_0);
		assert_same(long_binding_gen, long_value_gen_0, long_binding_gen, long_value_gen_0);
		assert_not_same(long_binding_ref, long_value_ref_0, long_binding_gen, long_value_gen_1);
		assert_not_same(long_binding_ref, long_value_ref_1, long_binding_gen, long_value_gen_0);		
		assert_not_same(long_binding_ref, long_value_ref_0, long_binding_ref, long_value_ref_1);
		assert_not_same(long_binding_gen, long_value_gen_0, long_binding_gen, long_value_gen_1);
		assert_valid(long_binding_ref, long_value_ref_0);
		assert_valid(long_binding_ref, long_value_ref_1);
		assert_valid(long_binding_gen, long_value_gen_0);
		assert_valid(long_binding_gen, long_value_gen_1);
	}

	// float
	@Test public void test_float() throws Exception {
		assert_same(float_binding_ref, float_value_ref_0, float_binding_gen, float_value_gen_0);
		assert_same(float_binding_ref, float_value_ref_1, float_binding_gen, float_value_gen_1);
		assert_same(float_binding_ref, float_value_ref_0, float_binding_ref, float_value_ref_0);
		assert_same(float_binding_gen, float_value_gen_0, float_binding_gen, float_value_gen_0);
		assert_not_same(float_binding_ref, float_value_ref_0, float_binding_gen, float_value_gen_1);
		assert_not_same(float_binding_ref, float_value_ref_1, float_binding_gen, float_value_gen_0);		
		assert_not_same(float_binding_ref, float_value_ref_0, float_binding_ref, float_value_ref_1);
		assert_not_same(float_binding_gen, float_value_gen_0, float_binding_gen, float_value_gen_1);
		assert_valid(float_binding_ref, float_value_ref_0);
		assert_valid(float_binding_ref, float_value_ref_1);
		assert_valid(float_binding_gen, float_value_gen_0);
		assert_valid(float_binding_gen, float_value_gen_1);
	}

	// double
	@Test public void test_double() throws Exception {
		assert_same(double_binding_ref, double_value_ref_0, double_binding_gen, double_value_gen_0);
		assert_same(double_binding_ref, double_value_ref_1, double_binding_gen, double_value_gen_1);
		assert_same(double_binding_ref, double_value_ref_0, double_binding_ref, double_value_ref_0);
		assert_same(double_binding_gen, double_value_gen_0, double_binding_gen, double_value_gen_0);
		assert_not_same(double_binding_ref, double_value_ref_0, double_binding_gen, double_value_gen_1);
		assert_not_same(double_binding_ref, double_value_ref_1, double_binding_gen, double_value_gen_0);		
		assert_not_same(double_binding_ref, double_value_ref_0, double_binding_ref, double_value_ref_1);
		assert_not_same(double_binding_gen, double_value_gen_0, double_binding_gen, double_value_gen_1);
		assert_valid(double_binding_ref, double_value_ref_0);
		assert_valid(double_binding_ref, double_value_ref_1);
		assert_valid(double_binding_gen, double_value_gen_0);
		assert_valid(double_binding_gen, double_value_gen_1);
	}

	// string
	@Test public void test_string() throws Exception {
		assert_same(string_binding_ref, string_value_ref_0, string_binding_gen, string_value_gen_0);
		assert_same(string_binding_ref, string_value_ref_1, string_binding_gen, string_value_gen_1);
		assert_same(string_binding_ref, string_value_ref_0, string_binding_ref, string_value_ref_0);
		assert_same(string_binding_gen, string_value_gen_0, string_binding_gen, string_value_gen_0);
		assert_not_same(string_binding_ref, string_value_ref_0, string_binding_gen, string_value_gen_1);
		assert_not_same(string_binding_ref, string_value_ref_1, string_binding_gen, string_value_gen_0);		
		assert_not_same(string_binding_ref, string_value_ref_0, string_binding_ref, string_value_ref_1);
		assert_not_same(string_binding_gen, string_value_gen_0, string_binding_gen, string_value_gen_1);
		assert_valid(string_binding_ref, string_value_ref_0);
		assert_valid(string_binding_ref, string_value_ref_1);
		assert_valid(string_binding_gen, string_value_gen_0);
		assert_valid(string_binding_gen, string_value_gen_1);
	}

	// array
	@Test public void test_array() throws Exception {
		assert_same(array_binding_ref, array_value_ref_0, array_binding_gen, array_value_gen_0);
		assert_same(array_binding_ref, array_value_ref_1, array_binding_gen, array_value_gen_1);
		assert_same(array_binding_ref, array_value_ref_0, array_binding_ref, array_value_ref_0);
		assert_same(array_binding_gen, array_value_gen_0, array_binding_gen, array_value_gen_0);
		assert_not_same(array_binding_ref, array_value_ref_0, array_binding_gen, array_value_gen_1);
		assert_not_same(array_binding_ref, array_value_ref_1, array_binding_gen, array_value_gen_0);		
		assert_not_same(array_binding_ref, array_value_ref_0, array_binding_ref, array_value_ref_1);
		assert_not_same(array_binding_gen, array_value_gen_0, array_binding_gen, array_value_gen_1);
		assert_valid(array_binding_ref, array_value_ref_0);
		assert_valid(array_binding_ref, array_value_ref_1);
		assert_valid(array_binding_gen, array_value_gen_0);
		assert_valid(array_binding_gen, array_value_gen_1);
	}

	// boolean_array
	@Test public void test_boolean_array() throws Exception {
		assert_same(boolean_array_binding_ref, boolean_array_value_ref_0, boolean_array_binding_gen, boolean_array_value_gen_0);
		assert_same(boolean_array_binding_ref, boolean_array_value_ref_1, boolean_array_binding_gen, boolean_array_value_gen_1);
		assert_same(boolean_array_binding_ref, boolean_array_value_ref_0, boolean_array_binding_ref, boolean_array_value_ref_0);
		assert_same(boolean_array_binding_gen, boolean_array_value_gen_0, boolean_array_binding_gen, boolean_array_value_gen_0);
		assert_not_same(boolean_array_binding_ref, boolean_array_value_ref_0, boolean_array_binding_gen, boolean_array_value_gen_1);
		assert_not_same(boolean_array_binding_ref, boolean_array_value_ref_1, boolean_array_binding_gen, boolean_array_value_gen_0);		
		assert_not_same(boolean_array_binding_ref, boolean_array_value_ref_0, boolean_array_binding_ref, boolean_array_value_ref_1);
		assert_not_same(boolean_array_binding_gen, boolean_array_value_gen_0, boolean_array_binding_gen, boolean_array_value_gen_1);
		assert_valid(boolean_array_binding_ref, boolean_array_value_ref_0);
		assert_valid(boolean_array_binding_ref, boolean_array_value_ref_1);
		assert_valid(boolean_array_binding_gen, boolean_array_value_gen_0);
		assert_valid(boolean_array_binding_gen, boolean_array_value_gen_1);
	}

	// byte_array
	@Test public void test_byte_array() throws Exception {
		assert_same(byte_array_binding_ref, byte_array_value_ref_0, byte_array_binding_gen, byte_array_value_gen_0);
		assert_same(byte_array_binding_ref, byte_array_value_ref_1, byte_array_binding_gen, byte_array_value_gen_1);
		assert_same(byte_array_binding_ref, byte_array_value_ref_0, byte_array_binding_ref, byte_array_value_ref_0);
		assert_same(byte_array_binding_gen, byte_array_value_gen_0, byte_array_binding_gen, byte_array_value_gen_0);
		assert_not_same(byte_array_binding_ref, byte_array_value_ref_0, byte_array_binding_gen, byte_array_value_gen_1);
		assert_not_same(byte_array_binding_ref, byte_array_value_ref_1, byte_array_binding_gen, byte_array_value_gen_0);		
		assert_not_same(byte_array_binding_ref, byte_array_value_ref_0, byte_array_binding_ref, byte_array_value_ref_1);
		assert_not_same(byte_array_binding_gen, byte_array_value_gen_0, byte_array_binding_gen, byte_array_value_gen_1);
		assert_valid(byte_array_binding_ref, byte_array_value_ref_0);
		assert_valid(byte_array_binding_ref, byte_array_value_ref_1);
		assert_valid(byte_array_binding_gen, byte_array_value_gen_0);
		assert_valid(byte_array_binding_gen, byte_array_value_gen_1);
	}

	// int_array
	@Test public void test_int_array() throws Exception {
		assert_same(int_array_binding_ref, int_array_value_ref_0, int_array_binding_gen, int_array_value_gen_0);
		assert_same(int_array_binding_ref, int_array_value_ref_1, int_array_binding_gen, int_array_value_gen_1);
		assert_same(int_array_binding_ref, int_array_value_ref_0, int_array_binding_ref, int_array_value_ref_0);
		assert_same(int_array_binding_gen, int_array_value_gen_0, int_array_binding_gen, int_array_value_gen_0);
		assert_not_same(int_array_binding_ref, int_array_value_ref_0, int_array_binding_gen, int_array_value_gen_1);
		assert_not_same(int_array_binding_ref, int_array_value_ref_1, int_array_binding_gen, int_array_value_gen_0);		
		assert_not_same(int_array_binding_ref, int_array_value_ref_0, int_array_binding_ref, int_array_value_ref_1);
		assert_not_same(int_array_binding_gen, int_array_value_gen_0, int_array_binding_gen, int_array_value_gen_1);
		assert_valid(int_array_binding_ref, int_array_value_ref_0);
		assert_valid(int_array_binding_ref, int_array_value_ref_1);
		assert_valid(int_array_binding_gen, int_array_value_gen_0);
		assert_valid(int_array_binding_gen, int_array_value_gen_1);
	}
	
	// int_array_2d
	@Test public void test_int_array_2d() throws Exception {
		assert_same(int_array_2d_binding_ref, int_array_2d_value_ref_0, int_array_2d_binding_gen, int_array_2d_value_gen_0);
		assert_same(int_array_2d_binding_ref, int_array_2d_value_ref_1, int_array_2d_binding_gen, int_array_2d_value_gen_1);
		assert_same(int_array_2d_binding_ref, int_array_2d_value_ref_0, int_array_2d_binding_ref, int_array_2d_value_ref_0);
		assert_same(int_array_2d_binding_gen, int_array_2d_value_gen_0, int_array_2d_binding_gen, int_array_2d_value_gen_0);
		assert_not_same(int_array_2d_binding_ref, int_array_2d_value_ref_0, int_array_2d_binding_gen, int_array_2d_value_gen_1);
		assert_not_same(int_array_2d_binding_ref, int_array_2d_value_ref_1, int_array_2d_binding_gen, int_array_2d_value_gen_0);		
		assert_not_same(int_array_2d_binding_ref, int_array_2d_value_ref_0, int_array_2d_binding_ref, int_array_2d_value_ref_1);
		assert_not_same(int_array_2d_binding_gen, int_array_2d_value_gen_0, int_array_2d_binding_gen, int_array_2d_value_gen_1);
		assert_valid(int_array_2d_binding_ref, int_array_2d_value_ref_0);
		assert_valid(int_array_2d_binding_ref, int_array_2d_value_ref_1);
		assert_valid(int_array_2d_binding_gen, int_array_2d_value_gen_0);
		assert_valid(int_array_2d_binding_gen, int_array_2d_value_gen_1);
	}	

	// long_array
	@Test public void test_long_array() throws Exception {
		assert_same(long_array_binding_ref, long_array_value_ref_0, long_array_binding_gen, long_array_value_gen_0);
		assert_same(long_array_binding_ref, long_array_value_ref_1, long_array_binding_gen, long_array_value_gen_1);
		assert_same(long_array_binding_ref, long_array_value_ref_0, long_array_binding_ref, long_array_value_ref_0);
		assert_same(long_array_binding_gen, long_array_value_gen_0, long_array_binding_gen, long_array_value_gen_0);
		assert_not_same(long_array_binding_ref, long_array_value_ref_0, long_array_binding_gen, long_array_value_gen_1);
		assert_not_same(long_array_binding_ref, long_array_value_ref_1, long_array_binding_gen, long_array_value_gen_0);		
		assert_not_same(long_array_binding_ref, long_array_value_ref_0, long_array_binding_ref, long_array_value_ref_1);
		assert_not_same(long_array_binding_gen, long_array_value_gen_0, long_array_binding_gen, long_array_value_gen_1);
		assert_valid(long_array_binding_ref, long_array_value_ref_0);
		assert_valid(long_array_binding_ref, long_array_value_ref_1);
		assert_valid(long_array_binding_gen, long_array_value_gen_0);
		assert_valid(long_array_binding_gen, long_array_value_gen_1);
	}

	// float_array
	@Test public void test_float_array() throws Exception {
		assert_same(float_array_binding_ref, float_array_value_ref_0, float_array_binding_gen, float_array_value_gen_0);
		assert_same(float_array_binding_ref, float_array_value_ref_1, float_array_binding_gen, float_array_value_gen_1);
		assert_same(float_array_binding_ref, float_array_value_ref_0, float_array_binding_ref, float_array_value_ref_0);
		assert_same(float_array_binding_gen, float_array_value_gen_0, float_array_binding_gen, float_array_value_gen_0);
		assert_not_same(float_array_binding_ref, float_array_value_ref_0, float_array_binding_gen, float_array_value_gen_1);
		assert_not_same(float_array_binding_ref, float_array_value_ref_1, float_array_binding_gen, float_array_value_gen_0);		
		assert_not_same(float_array_binding_ref, float_array_value_ref_0, float_array_binding_ref, float_array_value_ref_1);
		assert_not_same(float_array_binding_gen, float_array_value_gen_0, float_array_binding_gen, float_array_value_gen_1);
		assert_valid(float_array_binding_ref, float_array_value_ref_0);
		assert_valid(float_array_binding_ref, float_array_value_ref_1);
		assert_valid(float_array_binding_gen, float_array_value_gen_0);
		assert_valid(float_array_binding_gen, float_array_value_gen_1);
	}
	
	// double_array
	@Test public void test_double_array() throws Exception {
		assert_same(double_array_binding_ref, double_array_value_ref_0, double_array_binding_gen, double_array_value_gen_0);
		assert_same(double_array_binding_ref, double_array_value_ref_1, double_array_binding_gen, double_array_value_gen_1);
		assert_same(double_array_binding_ref, double_array_value_ref_0, double_array_binding_ref, double_array_value_ref_0);
		assert_same(double_array_binding_gen, double_array_value_gen_0, double_array_binding_gen, double_array_value_gen_0);
		assert_not_same(double_array_binding_ref, double_array_value_ref_0, double_array_binding_gen, double_array_value_gen_1);
		assert_not_same(double_array_binding_ref, double_array_value_ref_1, double_array_binding_gen, double_array_value_gen_0);		
		assert_not_same(double_array_binding_ref, double_array_value_ref_0, double_array_binding_ref, double_array_value_ref_1);
		assert_not_same(double_array_binding_gen, double_array_value_gen_0, double_array_binding_gen, double_array_value_gen_1);
		assert_valid(double_array_binding_ref, double_array_value_ref_0);
		assert_valid(double_array_binding_ref, double_array_value_ref_1);
		assert_valid(double_array_binding_gen, double_array_value_gen_0);
		assert_valid(double_array_binding_gen, double_array_value_gen_1);
	}

	// string_array
	@Test public void test_string_array() throws Exception {
		assert_same(string_array_binding_ref, string_array_value_ref_0, string_array_binding_gen, string_array_value_gen_0);
		assert_same(string_array_binding_ref, string_array_value_ref_1, string_array_binding_gen, string_array_value_gen_1);
		assert_same(string_array_binding_ref, string_array_value_ref_0, string_array_binding_ref, string_array_value_ref_0);
		assert_same(string_array_binding_gen, string_array_value_gen_0, string_array_binding_gen, string_array_value_gen_0);
		assert_not_same(string_array_binding_ref, string_array_value_ref_0, string_array_binding_gen, string_array_value_gen_1);
		assert_not_same(string_array_binding_ref, string_array_value_ref_1, string_array_binding_gen, string_array_value_gen_0);		
		assert_not_same(string_array_binding_ref, string_array_value_ref_0, string_array_binding_ref, string_array_value_ref_1);
		assert_not_same(string_array_binding_gen, string_array_value_gen_0, string_array_binding_gen, string_array_value_gen_1);
		assert_valid(string_array_binding_ref, string_array_value_ref_0);
		assert_valid(string_array_binding_ref, string_array_value_ref_1);
		assert_valid(string_array_binding_gen, string_array_value_gen_0);
		assert_valid(string_array_binding_gen, string_array_value_gen_1);
	}

	// record
	@Test public void test_record() throws Exception {
		assert_same(record_binding_ref, record_value_ref_0, record_binding_gen, record_value_gen_0);
		assert_same(record_binding_ref, record_value_ref_1, record_binding_gen, record_value_gen_1);
		assert_same(record_binding_ref, record_value_ref_0, record_binding_ref, record_value_ref_0);
		assert_same(record_binding_gen, record_value_gen_0, record_binding_gen, record_value_gen_0);
		assert_not_same(record_binding_ref, record_value_ref_0, record_binding_gen, record_value_gen_1);
		assert_not_same(record_binding_ref, record_value_ref_1, record_binding_gen, record_value_gen_0);		
		assert_not_same(record_binding_ref, record_value_ref_0, record_binding_ref, record_value_ref_1);
		assert_not_same(record_binding_gen, record_value_gen_0, record_binding_gen, record_value_gen_1);
		assert_valid(record_binding_ref, record_value_ref_0);
		assert_valid(record_binding_ref, record_value_ref_1);
		assert_valid(record_binding_gen, record_value_gen_0);
		assert_valid(record_binding_gen, record_value_gen_1);
	}

	// map
	@Test public void test_map() throws Exception {
		assert_same(map_binding_ref, map_value_ref_0, map_binding_gen, map_value_gen_0);
		assert_same(map_binding_ref, map_value_ref_1, map_binding_gen, map_value_gen_1);
		assert_same(map_binding_ref, map_value_ref_0, map_binding_ref, map_value_ref_0);
		assert_same(map_binding_gen, map_value_gen_0, map_binding_gen, map_value_gen_0);
		assert_not_same(map_binding_ref, map_value_ref_0, map_binding_gen, map_value_gen_1);
		assert_not_same(map_binding_ref, map_value_ref_1, map_binding_gen, map_value_gen_0);		
		assert_not_same(map_binding_ref, map_value_ref_0, map_binding_ref, map_value_ref_1);
		assert_not_same(map_binding_gen, map_value_gen_0, map_binding_gen, map_value_gen_1);
		assert_valid(map_binding_ref, map_value_ref_0);
		assert_valid(map_binding_ref, map_value_ref_1);
		assert_valid(map_binding_gen, map_value_gen_0);
		assert_valid(map_binding_gen, map_value_gen_1);
		
		test_ordered_map_binding(map_binding_ref);
		test_ordered_map_binding(map_binding_gen);		
	}

	// optional
	@Test public void test_optional() throws Exception {
		assert_same(optional_binding_ref, optional_value_ref_0, optional_binding_gen, optional_value_gen_0);
		assert_same(optional_binding_ref, optional_value_ref_1, optional_binding_gen, optional_value_gen_1);
		assert_same(optional_binding_ref, optional_value_ref_0, optional_binding_ref, optional_value_ref_0);
		assert_same(optional_binding_gen, optional_value_gen_0, optional_binding_gen, optional_value_gen_0);
		assert_not_same(optional_binding_ref, optional_value_ref_0, optional_binding_gen, optional_value_gen_1);
		assert_not_same(optional_binding_ref, optional_value_ref_1, optional_binding_gen, optional_value_gen_0);		
		assert_not_same(optional_binding_ref, optional_value_ref_0, optional_binding_ref, optional_value_ref_1);
		assert_not_same(optional_binding_gen, optional_value_gen_0, optional_binding_gen, optional_value_gen_1);
		assert_valid(optional_binding_ref, optional_value_ref_0);
		assert_valid(optional_binding_ref, optional_value_ref_1);
		assert_valid(optional_binding_gen, optional_value_gen_0);
		assert_valid(optional_binding_gen, optional_value_gen_1);
	}

	// union
	@Test public void test_union() throws Exception {
		assert_same(union_binding_ref, union_value_ref_0, union_binding_gen, union_value_gen_0);
		assert_same(union_binding_ref, union_value_ref_1, union_binding_gen, union_value_gen_1);
		assert_same(union_binding_ref, union_value_ref_0, union_binding_ref, union_value_ref_0);
		assert_same(union_binding_gen, union_value_gen_0, union_binding_gen, union_value_gen_0);
		assert_not_same(union_binding_ref, union_value_ref_0, union_binding_gen, union_value_gen_1);
		assert_not_same(union_binding_ref, union_value_ref_1, union_binding_gen, union_value_gen_0);		
		assert_not_same(union_binding_ref, union_value_ref_0, union_binding_ref, union_value_ref_1);
		assert_not_same(union_binding_gen, union_value_gen_0, union_binding_gen, union_value_gen_1);
		assert_valid(union_binding_ref, union_value_ref_0);
		assert_valid(union_binding_ref, union_value_ref_1);
		assert_valid(union_binding_gen, union_value_gen_0);
		assert_valid(union_binding_gen, union_value_gen_1);
	}
	
	// variant
	@Test public void test_variant() throws Exception {
		assert_same(variant_binding_ref, variant_value_ref_0, variant_binding_gen, variant_value_gen_0);
		assert_same(variant_binding_ref, variant_value_ref_1, variant_binding_gen, variant_value_gen_1);
		assert_same(variant_binding_ref, variant_value_ref_0, variant_binding_ref, variant_value_ref_0);
		assert_same(variant_binding_gen, variant_value_gen_0, variant_binding_gen, variant_value_gen_0);
		assert_not_same(variant_binding_ref, variant_value_ref_0, variant_binding_gen, variant_value_gen_1);
		assert_not_same(variant_binding_ref, variant_value_ref_1, variant_binding_gen, variant_value_gen_0);		
		assert_not_same(variant_binding_ref, variant_value_ref_0, variant_binding_ref, variant_value_ref_1);
		assert_not_same(variant_binding_gen, variant_value_gen_0, variant_binding_gen, variant_value_gen_1);
		assert_valid(variant_binding_ref, variant_value_ref_0);
		assert_valid(variant_binding_ref, variant_value_ref_1);
		assert_valid(variant_binding_gen, variant_value_gen_0);
		assert_valid(variant_binding_gen, variant_value_gen_1);
	}

	// datatype
	@Test public void test_datatype() throws Exception {
		assert_same(datatype_binding_ref, datatype_value_ref_0, datatype_binding_ref, datatype_value_ref_0);
		assert_not_same(datatype_binding_ref, datatype_value_ref_0, datatype_binding_ref, datatype_value_ref_1);
		assert_valid(datatype_binding_ref, datatype_value_ref_1);

		RandomValue rv = new RandomValue();
		rv.refereableRecords = true;
		
		int iteration_count = 100;
		for (int iteration = 0; iteration < iteration_count; iteration++) {
			int seed = iteration;
			rv.getRandom().setSeed(seed);
			rv.getRandom().nextLong();			
			Binding b = datatype_binding_ref;
			Object rt = rv.randomType(0, 3);
			Serializer s = b.serializer();
			byte[] bytes = s.serialize( rt );
			Object rt2 = s.deserialize( bytes );
			b.assertInstaceIsValid(rt2);
		}
		
	}
	
	// variant
	@Test public void test_random_values() throws Exception {
		int iteration_count = 10000;
		
		RandomValue rv = new RandomValue();
		rv.refereableRecords = true;
		for (int iteration = 0; iteration < iteration_count; iteration++) {			
			int seed = iteration;
			rv.getRandom().setSeed(seed);
			rv.getRandom().nextLong();
			
			Object variant_value_gen_random_0 = variant_binding_gen.accept( rv );
			
			rv.getRandom().setSeed(seed);
			rv.getRandom().nextLong();			
			Object variant_value_gen_random_1 = variant_binding_gen.accept( rv );
			
			assert_same(variant_binding_gen, variant_value_gen_random_0, variant_binding_gen, variant_value_gen_random_0);
			assert_same(variant_binding_gen, variant_value_gen_random_1, variant_binding_gen, variant_value_gen_random_1);
			assert_valid(variant_binding_gen, variant_value_gen_random_0);
			assert_valid(variant_binding_gen, variant_value_gen_random_1);
		}
	}	
	
	static void assert_same(Binding b1, Object o1, Binding b2, Object o2) throws BindingException, SerializationException, IOException, DataTypeSyntaxError, SerializerConstructionException
	{
		// DataType Test
		assertEquals( b1.type(), b2.type() );
		Datatype t = b1.type();
		
		// CompareTo Test
		assertEquals(0, Bindings.compare(b1, o1, b2, o2) );
		
		// Equals Test
		assertTrue( Bindings.equals(b1, o1, b2, o2) );
		
		// HashCode Test
		int hash1 = b1.hashValue(o1);
		int hash2 = b2.hashValue(o2);
		assertEquals( hash1, hash2 );

		// Validator Test
		b1.assertInstaceIsValid(o1);
		b2.assertInstaceIsValid(o2);
		
		// Default Value Test
		Object x1 = b1.createDefault();
		Object x2 = b2.createDefault();
		b1.assertInstaceIsValid(x1);
		b2.assertInstaceIsValid(x2);
		assertEquals(0, Bindings.compare(b1, x1, b2, x2) );
		assertTrue( Bindings.equals(b1, x1, b2, x2) );
				
		// Clone Test
		Object o1_ = Bindings.cloneUnchecked(o2, b2, b1);
		Object o2_ = Bindings.cloneUnchecked(o1, b1, b2);
		if (!b1.isImmutable() && !b2.isImmutable()) { 
			assertTrue(o1!=o1_);
			assertTrue(o2!=o2_);
		}
		assertEquals(0, b1.compare(o1, o1_));
		assertEquals(0, b2.compare(o2, o2_));
		assertTrue( b1.equals(o1, o1_) );		
		assertTrue( b2.equals(o2, o2_) );				
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);
		
		o1_ = b1.cloneUnchecked(o1);
		o2_ = b2.cloneUnchecked(o2);
		if (!b1.isImmutable()) assertTrue(o1!=o1_);
		if (!b2.isImmutable()) assertTrue(o2!=o2_);
		assertEquals(0, b1.compare(o1, o1_));
		assertEquals(0, b2.compare(o2, o2_));
		assertTrue( b1.equals(o1, o1_) );		
		assertTrue( b2.equals(o2, o2_) );				
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);
		
		
		// Adapter Test
		o1_ = Bindings.adaptUnchecked(o2, b2, b1);
		o2_ = Bindings.adaptUnchecked(o1, b1, b2);
		
		assertEquals(0, b1.compare(o1, o1_));
		assertEquals(0, b2.compare(o2, o2_));
		assertTrue( b1.equals(o1, o1_) );		
		assertTrue( b2.equals(o2, o2_) );				
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);
		
		// Binary Serializer Test
		Serializer s1 = Bindings.getSerializer( b1 );
		Serializer s2 = Bindings.getSerializer( b2 );
		byte[] byte1 = s1.serialize(o1);
		byte[] byte2 = s2.serialize(o2);
		assertNotNull(byte1);
		assertNotNull(byte2);
		assertTrue( Arrays.equals(byte1, byte2) );
		o1_ = s1.deserialize( byte2 );
		o2_ = s2.deserialize( byte1 );

		assertEquals(0, b1.compare(o1, o1_));
		assertTrue( b1.equals(o1, o1_) );
		assertEquals(0, b2.compare(o2, o2_));
		assertTrue( b2.equals(o2, o2_) );
				
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);
				
		// Ascii Serializer Test
		if (!OMIT_PARSER_TEST) {
		String str1 = b1.printValueDefinition(o1, true);
		String str2 = b2.printValueDefinition(o2, true);
		assertEquals(str1, str2);
		o1_ = b1.parseValueDefinition(str1);
		o2_ = b2.parseValueDefinition(str2);		
		assertEquals(0, b1.compare(o1, o1_));
		assertEquals(0, b2.compare(o2, o2_));
		assertTrue( b1.equals(o1, o1_) );
		assertTrue( b2.equals(o2, o2_) );
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);		
		
		str1 = b1.printValueDefinition(o1, false);
		str2 = b2.printValueDefinition(o2, false);
		o1_ = b1.parseValueDefinition(str1);
		o2_ = b2.parseValueDefinition(str2);		
		assertEquals(0, b1.compare(o1, o1_));
		assertTrue( b1.equals(o1, o1_) );
		assertEquals(0, b2.compare(o2, o2_));
		assertTrue( b2.equals(o2, o2_) );
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);
		}

	}	
	
	static void assert_not_same(Binding b1, Object o1, Binding b2, Object o2) throws BindingException, SerializationException, IOException, DataTypeSyntaxError, SerializerConstructionException
	{
		// DataType Test
		assertEquals( b1.type(), b2.type() );
		Datatype t = b1.type();
		
		// CompareTo Test
		int dif = Integer.signum(  Bindings.compare(b1, o1, b2, o2)  );
		int dif2 = Integer.signum(  Bindings.compare(b2, o2, b1, o1)  );
		assertEquals(-dif2, dif);
		
		// Equals Test
		assertFalse( Bindings.equals(b1, o1, b2, o2) );
		
		// HashCode Test (they could match by coincidence, we trust they don't)
		int hash1 = b1.hashValue(o1);
		int hash2 = b2.hashValue(o2);
		assertNotSame( hash1, hash2 );		
		
		// Validator Test
		b1.assertInstaceIsValid(o1);
		b2.assertInstaceIsValid(o2);
		
		// Adapter Test
		Object o1_ = Bindings.adaptUnchecked(o2, b2, b1);
		Object o2_ = Bindings.adaptUnchecked(o1, b1, b2);
		dif = Integer.signum(  Bindings.compare(b1, o1_, b2, o2_)  );
		dif2 = Integer.signum(  Bindings.compare(b2, o2_, b1, o1_)  );
		assertEquals(-dif2, dif);
		assertFalse( b1.equals(o1, o1_) );		
		assertFalse( b2.equals(o2, o2_) );		
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);

		// Binary Serializer Test
		Serializer s1 = Bindings.getSerializer( b1 );
		Serializer s2 = Bindings.getSerializer( b2 );
		byte[] byte1 = s1.serialize(o1);
		byte[] byte2 = s2.serialize(o2);
		assertFalse( Arrays.equals(byte1, byte2) );
		o1_ = s1.deserialize( byte2 );
		o2_ = s2.deserialize( byte1 );
		dif = Integer.signum(  Bindings.compare(b1, o1_, b2, o2_)  );
		dif2 = Integer.signum(  Bindings.compare(b2, o2_, b1, o1_)  );
		assertEquals(-dif2, dif);
		dif = Integer.signum(  Bindings.compare(b1, o1, b1, o1_)  );
		dif2 = Integer.signum(  Bindings.compare(b2, o2, b2, o2_)  );
		assertEquals(-dif2, dif);
		assertFalse( b1.equals(o1, o1_) );		
		assertFalse( b2.equals(o2, o2_) );		
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);

		
		
		// Ascii Serializer Test
		if (!OMIT_PARSER_TEST) {
		String str1 = b1.printValueDefinition(o1, true);
		String str2 = b2.printValueDefinition(o2, true);
		assertNotSame(str1, str2);
		o1_ = b1.parseValueDefinition(str2);
		o2_ = b2.parseValueDefinition(str1);		
		dif = Integer.signum(  Bindings.compare(b1, o1_, b2, o2_)  );
		dif2 = Integer.signum(  Bindings.compare(b2, o2_, b1, o1_)  );
		assertEquals(-dif2, dif);
		dif = Integer.signum(  Bindings.compare(b1, o1, b1, o1_)  );
		dif2 = Integer.signum(  Bindings.compare(b2, o2, b2, o2_)  );
		assertEquals(-dif2, dif);
		assertFalse( b1.equals(o1, o1_) );		
		assertFalse( b2.equals(o2, o2_) );		
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);
		
		str1 = b1.printValueDefinition(o1, false);
		str2 = b2.printValueDefinition(o2, false);
		assertNotSame(str1, str2);
		o1_ = b1.parseValueDefinition(str2);
		o2_ = b2.parseValueDefinition(str1);		
		dif = Integer.signum(  Bindings.compare(b1, o1_, b2, o2_)  );
		dif2 = Integer.signum(  Bindings.compare(b2, o2_, b1, o1_)  );
		assertEquals(-dif2, dif);
		dif = Integer.signum(  Bindings.compare(b1, o1, b1, o1_)  );
		dif2 = Integer.signum(  Bindings.compare(b2, o2, b2, o2_)  );
		assertEquals(-dif2, dif);
		assertFalse( b1.equals(o1, o1_) );		
		assertFalse( b2.equals(o2, o2_) );		
		b1.assertInstaceIsValid(o1_);
		b2.assertInstaceIsValid(o2_);	
		}
	}	
	
	static void assert_valid(Binding b, Object o) throws BindingException, IOException, SerializationException, DataTypeSyntaxError, SerializerConstructionException {
		// Validator test
		b.assertInstaceIsValid(o);
		assertTrue( b.isInstance(o) );

		// Serializer test
		Serializer s = Bindings.getSerializer( b );
		byte[] bytes = s.serialize(o);
		
		InputStream is = new ByteArrayInputStream(bytes);
		s.skip(is);
		assertEquals(0, is.available());
		
		is = new ByteArrayInputStream(bytes);
		s.deserialize(is);
		assertEquals(0, is.available());
		
		Object o_ = s.deserialize( bytes );
		assertEquals(0, b.compare(o, o_));		
		assertTrue( b.equals(o, o_) );
		
		// deserialize to instance Test
		Binding gb = Bindings.getMutableBinding( b.type() );
		o_ = gb.createDefault();
		s = gb.serializer();
		s.deserialize(bytes, o_);
		assertTrue(Bindings.equals(b, o, gb, o_));
		
		// Read from test
		o_ = gb.createDefault();
		gb.readFrom(b, o, o_);
		assertTrue(Bindings.equals(b, o, gb, o_));
		
		
		// Ascii Serializer Test
		if (!OMIT_PARSER_TEST) {
		String str = b.printValueDefinition(o, true);
		o_ = b.parseValueDefinition(str);
		assertEquals(0, b.compare(o, o_));		
		assertTrue( b.equals(o, o_) );
		b.assertInstaceIsValid(o_);
		assertTrue( b.isInstance(o_) );
		}
	}
	
	/**
	 * This case tests ordered map features getFirst, getLast, getLower, getFloor, getHigher, getCeiling 
	 * 
	 * @param valueBinding Map(Integer, String)
	 */
	void test_ordered_map_binding(MapBinding b) throws Exception {
		
		Datatype keyType = b.type().keyType;
		Datatype valueType = b.type().valueType;
		
		assertTrue(keyType instanceof IntegerType);
		assertTrue(valueType instanceof StringType);
		
		IntegerBinding kb = (IntegerBinding) b.getKeyBinding();
		
		Object map = b.createDefault();
		Object value = "";
		b.put(map, kb.create(0), value);
		b.put(map, kb.create(50), value);
		b.put(map, kb.create(10), value);
		b.put(map, kb.create(30), value);
		b.put(map, kb.create(40), value);
		b.put(map, kb.create(60), value);
		b.put(map, kb.create(20), value);

		assertEquals( b.getFirstKey(map), kb.create(0) );
		assertEquals( b.getLastKey(map), kb.create(60) );

		assertEquals( b.getFloorKey(map, kb.create(30)), kb.create(30));
		assertEquals( b.getFloorKey(map, kb.create(29)), kb.create(20));
		assertEquals( b.getFloorKey(map, kb.create(31)), kb.create(30));
		assertEquals( b.getFloorKey(map, kb.create(70)), kb.create(60));
		assertEquals( b.getFloorKey(map, kb.create(60)), kb.create(60));
		assertEquals( b.getFloorKey(map, kb.create(61)), kb.create(60));
		assertEquals( b.getFloorKey(map, kb.create(1)), kb.create(0));
		assertEquals( b.getFloorKey(map, kb.create(0)), kb.create(0));
		assertEquals( b.getFloorKey(map, kb.create(-1)), null);

		assertEquals( b.getLowerKey(map, kb.create(30)), kb.create(20));
		assertEquals( b.getLowerKey(map, kb.create(29)), kb.create(20));
		assertEquals( b.getLowerKey(map, kb.create(31)), kb.create(30));
		assertEquals( b.getLowerKey(map, kb.create(70)), kb.create(60));
		assertEquals( b.getLowerKey(map, kb.create(60)), kb.create(50));
		assertEquals( b.getLowerKey(map, kb.create(61)), kb.create(60));
		assertEquals( b.getLowerKey(map, kb.create(1)), kb.create(0));
		assertEquals( b.getLowerKey(map, kb.create(0)), null);
		assertEquals( b.getLowerKey(map, kb.create(-1)), null);
		
		assertEquals( b.getCeilingKey(map, kb.create(30)), kb.create(30));
		assertEquals( b.getCeilingKey(map, kb.create(29)), kb.create(30));
		assertEquals( b.getCeilingKey(map, kb.create(31)), kb.create(40));
		assertEquals( b.getCeilingKey(map, kb.create(-10)), kb.create(0));
		assertEquals( b.getCeilingKey(map, kb.create(60)), kb.create(60));
		assertEquals( b.getCeilingKey(map, kb.create(61)), null);

		assertEquals( b.getHigherKey(map, kb.create(30)), kb.create(40));
		assertEquals( b.getHigherKey(map, kb.create(29)), kb.create(30));
		assertEquals( b.getHigherKey(map, kb.create(31)), kb.create(40));		
		assertEquals( b.getHigherKey(map, kb.create(-10)), kb.create(0));
		assertEquals( b.getHigherKey(map, kb.create(60)), null);
		assertEquals( b.getHigherKey(map, kb.create(61)), null);
		
		
	}

}

