/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.TreeMap;

import junit.framework.TestCase;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.adapter.Adapter;
import org.simantics.databoard.adapter.AdapterConstructionException;
import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.MapBinding;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.VariantBinding;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.impl.DoubleBindingDefault;
import org.simantics.databoard.binding.impl.IntegerBindingDefault;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.IntegerType;

public class TestAdapter extends TestCase {
	
	
	
	/////////////////////////////////////////////
	public static enum FooEnum {Ford, Volvo, Audi, Mazda, Nissan, Honda, Toyota, BMW, MercedezBenz}
	public static enum BarEnum {Audi, BMW, MercedezBenz}	
	
	public void testUnionAdaption() throws Exception 
	{
		Binding fooB = Bindings.getBinding( FooEnum.class );
		Binding barB = Bindings.getBinding( BarEnum.class );
		
		assertEquals( FooEnum.Audi, Bindings.adapt( BarEnum.Audi, barB, fooB ) );
		assertEquals( FooEnum.MercedezBenz, Bindings.adapt( BarEnum.MercedezBenz, barB, fooB ) );
		assertEquals( FooEnum.BMW, Bindings.adapt( BarEnum.BMW, barB, fooB ) );

		// Fail-Test, We shouldn't be able to convert from Foo to Bar (from super-type to sub-type)
		try {
			assertEquals( BarEnum.BMW, Bindings.adapt( FooEnum.BMW, fooB, barB ) );
			fail();
		} catch (AdaptException e) {}
	}
	/////////////////////////////////////////////
	
	/////////////////////////////////////////////
	public static class Foo2 {
		public double x, y, z;
	}

	public static class Bar2 {
		public int id; 
		public float y, z, x;
	}
	
	public void testRecordAdaption() throws Exception
	{
		Binding fooB = Bindings.getBinding( Foo2.class );
		Binding barB = Bindings.getBinding( Bar2.class );
		Bar2 bar = new Bar2();
		bar.id = 5;
		bar.x = 5.0f;
		bar.y = 7.0f;
		bar.z = 123.43243244242423f;
		
		Foo2 foo = (Foo2) Bindings.adapt(bar, barB, fooB);
		assertEquals(bar.x, (float) foo.x);
		assertEquals(bar.y, (float) foo.y);
		assertEquals(bar.z, (float) foo.z);
		
		// Fail-Test, Cannot convert Bar2 to Foo2 because there is Id field
		// No longer fails for missing fields after r30952 (02/19/15, Niemist)
		/*
		try {
			Bindings.adapt(foo, fooB, barB);
			fail();
			
		} catch (AdaptException e) {			
		}
		*/		
	}
	/////////////////////////////////////////////
	
	
	
	public void testPrimitiveAdaptation() throws AdaptException, BindingConstructionException 
	{
		try {			
			Adapter int2Double = Bindings.getAdapter(Bindings.getBinding(int.class), Bindings.getBinding(double.class));
			fail("Exception excepted");
		} catch (AdapterConstructionException e) {			
		}
		
		try {			
			Adapter double2Double = Bindings.getAdapter(Bindings.getBinding(double.class), Bindings.getBinding(double.class));
			assertEquals(5.0, double2Double.adapt(5.0));
			
			Adapter int2Double = Bindings.getTypeAdapter(Bindings.getBinding(int.class), Bindings.getBinding(double.class));
			assertEquals(5.0, int2Double.adapt(5));
		} catch (AdapterConstructionException e) {			
			fail("Exception unexcepted");
		}
		
		try {			
			Adapter doubleArray2intArray = Bindings.getAdapter(Bindings.getBinding(double[].class), Bindings.getBinding(int[].class));
			assertEquals(new double[] {5.0, 6.0, 7.0}, doubleArray2intArray.adapt(new int[] {5, 6, 7}));			
			fail("Exception excepted");
		} catch (AdapterConstructionException e) {			
		}
		
		try {			
			Adapter doubleArray2intArray = Bindings.getTypeAdapter(Bindings.getBinding(double[].class), Bindings.getBinding(int[].class));
			assertTrue( Arrays.equals( new int[] {5, 6, 7}, (int[]) doubleArray2intArray.adapt(new double[] {5.0, 6.0, 7.0})) );			
		} catch (AdapterConstructionException e) {			
			fail("Exception unexcepted");
		}
		
		try {			
			Adapter doubleArray2intArray = Bindings.getTypeAdapter(Bindings.getBinding(Double[].class), Bindings.getBinding(int[].class));
			assertTrue( Arrays.equals( new int[] {5, 6, 7}, (int[]) doubleArray2intArray.adapt(new Double[] {5.0, 6.0, 7.0})) );			
		} catch (AdapterConstructionException e) {			
			fail("Exception unexcepted");
		}

		
	}

	public void testUnitAdaptation() throws AdaptException
	{
		try {
			Adapter hours2Seconds = Bindings.getAdapter(new DoubleBindingDefault(new DoubleType("h")), new DoubleBindingDefault(new DoubleType("s")));
			assertEquals(3600.0, hours2Seconds.adapt(1.0));
		} catch (AdapterConstructionException e) {
			fail("Exception unexcepted");
		}

		try {
			Adapter m2ft = Bindings.getAdapter(new DoubleBindingDefault(new DoubleType("m")), new DoubleBindingDefault(new DoubleType("ft")));
			assertTrue( Math.abs( ( (Double) m2ft.adapt(1.524) ) - 5.0 ) < 0.1 );
		} catch (AdapterConstructionException e) {
			fail("Exception unexcepted");
		}
		
		try {
			Adapter hours2Seconds = Bindings.getAdapter(new IntegerBindingDefault(new IntegerType("h")), new IntegerBindingDefault(new IntegerType("s")));
			assertEquals(3600, hours2Seconds.adapt(1));
		} catch (AdapterConstructionException e) {
			fail("Exception unexcepted");
		}
		
		try {
			Integer hours = 1;
			Double seconds = (Double) Bindings.adapt(hours, new IntegerBindingDefault(new IntegerType("h")), new DoubleBindingDefault(new DoubleType("s")));
			assertEquals(3600.0, seconds);
		} catch (AdaptException e) {
			fail("Exception unexcepted");
		}
				
	}
	
	public static class Foo {
		List<String> list;
		String[] array;
		LinkedList<String> linkedList;		
		HashMap<Integer, String> hashMap;
		TreeMap<Integer, String> treeMap;
	}
	
	@SuppressWarnings("unchecked")
	public void testListAdaption() throws Exception
	{
		RecordBinding fooBinding  = (RecordBinding) Bindings.getBinding(Foo.class);
		ArrayBinding listBinding  = (ArrayBinding) fooBinding.getComponentBindings()[0];
		ArrayBinding arrayBinding = (ArrayBinding) fooBinding.getComponentBindings()[1];
		ArrayBinding linkedListBinding = (ArrayBinding) fooBinding.getComponentBindings()[2];
		Adapter list2array = Bindings.getAdapter(listBinding, arrayBinding);
		Adapter array2linkedList = Bindings.getAdapter(arrayBinding, linkedListBinding);
		Foo foo = new Foo();
		foo.list = new ArrayList<String>();
		foo.list.add("XYZ");
		foo.array = (String[]) list2array.adapt(foo.list);
		foo.linkedList = (LinkedList<String>) array2linkedList.adapt(foo.array);
		assertEquals(foo.array[0], foo.list.get(0));
		assertEquals(foo.list, foo.linkedList);
		
		assertEquals( listBinding.hashValue(foo.list), arrayBinding.hashValue(foo.array) );
		assertEquals( listBinding.hashValue(foo.list), linkedListBinding.hashValue(foo.linkedList) );
	}
	
	@SuppressWarnings("unchecked")
	public void testMapAdaption() throws Exception
	{
		RecordBinding fooBinding  = (RecordBinding) Bindings.getBinding(Foo.class);
		MapBinding hashMapBinding = (MapBinding) fooBinding.getComponentBindings()[3];
		MapBinding treeMapBinding = (MapBinding) fooBinding.getComponentBindings()[4];
		Adapter list2array = Bindings.getAdapter(hashMapBinding, treeMapBinding);
		Foo foo = new Foo();
		foo.hashMap = new HashMap<Integer, String>();
		foo.hashMap.put(500, "Indy-500");
		foo.treeMap = (TreeMap<Integer, String>) list2array.adapt(foo.hashMap);
		assertEquals(foo.hashMap, foo.treeMap);
	}
	
	public void testVariantAdaption() throws Exception
	{
		VariantBinding variantBinding  = (VariantBinding) Bindings.getBinding(Object.class);
		VariantBinding variant2Binding = (VariantBinding) Bindings.getMutableBinding(Datatypes.VARIANT);
		
		Adapter obj2variant = Bindings.getAdapter(variantBinding, variant2Binding);
		Object genericVariant = obj2variant.adapt( "blondi" );
		System.out.println(genericVariant);		
	}
	
	
	
}

