/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.mapping.constraint.instructions;

import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.set.hash.TIntHashSet;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.mapping.IContextualModification;

public class TypedBracketInstruction implements IInstruction {

	public static abstract class CreationInstruction {
		protected int variableId;
		
		public CreationInstruction(int variableId) {
			this.variableId = variableId;
		}

		public abstract Resource create(WriteGraph g, Object[] bindings) throws DatabaseException;

		public void mapVariables(TIntIntHashMap map) {
			variableId = map.get(variableId);			
		}
		
		public void collectVariables(TIntHashSet reads, TIntHashSet writes) {			
		}
	}
	
	IInstruction instruction;
	CreationInstruction[] variables;	
	
	public TypedBracketInstruction(IInstruction instruction, CreationInstruction ... variables) {
		this.instruction = instruction;
		this.variables = variables;
	}	

	@Override
	public void collectVariables(TIntHashSet reads, TIntHashSet writes) {
		instruction.collectVariables(reads, writes);
		for(CreationInstruction var : variables)
			var.collectVariables(reads, writes);
	}
	
	class Claim implements IContextualModification {
			
		public Claim() {
		}

		@Override
		public void perform(WriteGraph g, Object[] bindings) throws DatabaseException {
			for(CreationInstruction var : variables) {
				bindings[var.variableId] = var.create(g, bindings);
				if(DEBUG)
				    System.out.println("@ newResource " + NameUtils.getSafeName(g, (Resource)bindings[var.variableId]));
			}
			instruction.doClaim(g, bindings);	
		}
	}
	
	@Override
	public IContextualModification claim(ReadGraph g, Object[] bindings) throws DatabaseException {
		if(instruction.query(g, bindings)==IInstruction.FAILURE)
			return new Claim();
		return null;
	}
	
	class Deny implements IContextualModification {
		Object continuation;
		
		public Deny(Object continuation) {
			this.continuation = continuation;
		}

		@Override
		public void perform(WriteGraph g, Object[] bindings) throws DatabaseException {
			do {
				for(CreationInstruction var : variables)
					g.deny((Resource)bindings[var.variableId]);
				continuation = instruction.next(g, bindings, continuation);
			} while(continuation != null && continuation != IInstruction.FAILURE);	
		}
	}
	
	@Override
	public IContextualModification deny(ReadGraph g, Object[] bindings) throws DatabaseException {
		Object continuation = instruction.query(g, bindings);
		if(continuation != IInstruction.FAILURE)
			return new Deny(continuation);
		return null;
	}

	@Override
	public void doClaim(WriteGraph g, Object[] bindings) throws DatabaseException {
		if(instruction.query(g, bindings)==IInstruction.FAILURE) {
			for(CreationInstruction var : variables) {
				bindings[var.variableId] = var.create(g, bindings);
				if(DEBUG)
				    System.out.println("@ newResource " + NameUtils.getSafeName(g, (Resource)bindings[var.variableId]));
			}
			instruction.doClaim(g, bindings);
		}		
	}

	@Override
	public void doDeny(WriteGraph g, Object[] bindings) throws DatabaseException {
		Object continuation = instruction.query(g, bindings);
		if(continuation != IInstruction.FAILURE)
			do {
				for(CreationInstruction var : variables)
					g.deny((Resource)bindings[var.variableId]);
				continuation = instruction.next(g, bindings, continuation);
			} while(continuation != null && continuation != IInstruction.FAILURE);		
	}

	@Override
	public void mapVariables(TIntIntHashMap map) {
		instruction.mapVariables(map);
		for(int i=0;i<variables.length;++i)
			variables[i].mapVariables(map);
	}

	@Override
	public Object next(ReadGraph g, Object[] bindings, Object continuation) throws DatabaseException {
		return instruction.next(g, bindings, continuation);
	}

	@Override
	public Object query(ReadGraph g, Object[] bindings) throws DatabaseException {
		return instruction.query(g, bindings);
	}

    @Override
    public void toString(StringBuilder b, int indent) {
        b.append('[');
        b.append(']');
    }

}
