/*******************************************************************************
 * Copyright (c) 2019 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.request;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.ValueAccessor;
import org.simantics.db.layer0.variable.VariableBuilder;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Pair;

public class PropertyInfo {
    public final Resource predicate;
    public final String name;
    public final boolean isImmutable;
    public final boolean isHasProperty;
    public final boolean isFunctional;
    public final Set<String> classifications;
    public final VariableBuilder builder;
    public final Resource literalRange;
    public final Datatype requiredDatatype;
    public final String requiredValueType;
    public final String definedUnit;
    public final Binding defaultBinding;
    public final Map<String,Pair<Resource, ChildReference>> subliteralPredicates;
    public final ValueAccessor valueAccessor;
    public final boolean hasEnumerationRange;
    public PropertyInfo(Resource predicate, String name, boolean isImmutable, boolean isFunctional, boolean isHasProperty, Set<String> classifications, VariableBuilder builder, Resource literalRange, Datatype requiredDatatype, String definedUnit, String requiredValueType, Binding defaultBinding, Map<String,Pair<Resource, ChildReference>> subliteralPredicates, ValueAccessor valueAccessor, boolean hasEnumerationRange) {
        this.predicate = predicate;
        this.name = name;
        this.isImmutable = isImmutable;
        this.isFunctional = isFunctional;
        this.isHasProperty = isHasProperty;
        this.classifications = classifications;
        this.builder = builder;
        this.literalRange = literalRange;
        this.requiredDatatype = requiredDatatype;
        this.definedUnit = definedUnit;
        this.requiredValueType = requiredValueType;
        this.defaultBinding = defaultBinding;
        this.subliteralPredicates = subliteralPredicates;
        this.valueAccessor = valueAccessor;
        this.hasEnumerationRange = hasEnumerationRange;
    }
    public static PropertyInfo make(ReadGraph graph, Resource predicate, String name, boolean isFunctional, boolean isHasProperty, Set<String> classifications, VariableBuilder builder, Resource literalRange, Datatype requiredDatatype, String definedUnit, String requiredValueType, Map<String,Pair<Resource, ChildReference>> subliteralPredicates, ValueAccessor valueAccessor, boolean hasEnumerationRange) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        if(literalRange != null) {
            Collection<Resource> dts = graph.getAssertedObjects(literalRange, L0.HasDataType);
            if(dts.size() == 1) {
                Datatype dt = graph.getPossibleValue(dts.iterator().next(), Bindings.DATATYPE);
                if(requiredDatatype == null) requiredDatatype = dt;
            }
        }
        Binding defaultBinding = requiredDatatype != null ? Bindings.getBinding(requiredDatatype) : null;
        return new PropertyInfo(predicate, name, graph.isImmutable(predicate), isFunctional, isHasProperty, classifications, builder, literalRange, requiredDatatype, definedUnit, requiredValueType, defaultBinding, subliteralPredicates, valueAccessor, hasEnumerationRange);
    }
    public boolean hasClassification(String classification) {
        return classifications.contains(classification);
    }
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("PropertyInfo [")
        .append(name)
        .append(" : ")
        .append(requiredDatatype)
        .append(" :: ")
        .append(requiredValueType)
        .append(", predicate=")
        .append(predicate)
        .append(", isFunctional=")
        .append(isFunctional)
        .append(", isHasProperty=")
        .append(isHasProperty)
        .append(", hasEnumerationRange=")
        .append(hasEnumerationRange)
        .append(", definedUnit=")
        .append(definedUnit != null ? definedUnit : "<none>")
        .append(", defaultBinding=")
        .append(defaultBinding)
        .append(", valueAccessor=")
        .append(valueAccessor)
        .append("]");
        return sb.toString();
    }
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((predicate == null) ? 0 : predicate.hashCode());
        if (!isImmutable) {
            result = prime * result + ((builder == null) ? 0 : builder.hashCode());
            result = prime * result + ((classifications == null) ? 0 : classifications.hashCode());
            result = prime * result + ((defaultBinding == null) ? 0 : defaultBinding.hashCode());
            result = prime * result + ((definedUnit == null) ? 0 : definedUnit.hashCode());
            result = prime * result + (hasEnumerationRange ? 1231 : 1237);
            result = prime * result + (isFunctional ? 1231 : 1237);
            result = prime * result + (isHasProperty ? 1231 : 1237);
            result = prime * result + ((literalRange == null) ? 0 : literalRange.hashCode());
            result = prime * result + ((name == null) ? 0 : name.hashCode());
            result = prime * result + ((requiredDatatype == null) ? 0 : requiredDatatype.hashCode());
            result = prime * result + ((requiredValueType == null) ? 0 : requiredValueType.hashCode());
            result = prime * result + ((subliteralPredicates == null) ? 0 : subliteralPredicates.hashCode());
            result = prime * result + ((valueAccessor == null) ? 0 : valueAccessor.hashCode());
        }
        return result;
    }
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        PropertyInfo other = (PropertyInfo) obj;
        if (predicate == null) {
            if (other.predicate != null)
                return false;
        } else if (!predicate.equals(other.predicate))
            return false;
        if(isImmutable)
            return true;
        if (builder == null) {
            if (other.builder != null)
                return false;
        } else if (!builder.equals(other.builder))
            return false;
        if (classifications == null) {
            if (other.classifications != null)
                return false;
        } else if (!classifications.equals(other.classifications))
            return false;
        if (defaultBinding == null) {
            if (other.defaultBinding != null)
                return false;
        } else if (!defaultBinding.equals(other.defaultBinding))
            return false;
        if (definedUnit == null) {
            if (other.definedUnit != null)
                return false;
        } else if (!definedUnit.equals(other.definedUnit))
            return false;
        if (hasEnumerationRange != other.hasEnumerationRange)
            return false;
        if (isFunctional != other.isFunctional)
            return false;
        if (isHasProperty != other.isHasProperty)
            return false;
        if (literalRange == null) {
            if (other.literalRange != null)
                return false;
        } else if (!literalRange.equals(other.literalRange))
            return false;
        if (name == null) {
            if (other.name != null)
                return false;
        } else if (!name.equals(other.name))
            return false;
        if (requiredDatatype == null) {
            if (other.requiredDatatype != null)
                return false;
        } else if (!requiredDatatype.equals(other.requiredDatatype))
            return false;
        if (requiredValueType == null) {
            if (other.requiredValueType != null)
                return false;
        } else if (!requiredValueType.equals(other.requiredValueType))
            return false;
        if (subliteralPredicates == null) {
            if (other.subliteralPredicates != null)
                return false;
        } else if (!subliteralPredicates.equals(other.subliteralPredicates))
            return false;
        if (valueAccessor == null) {
            if (other.valueAccessor != null)
                return false;
        } else if (!valueAccessor.equals(other.valueAccessor))
            return false;
        return true;
    }
}
