/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.map;

import com.koloboke.collect.Container;
import com.koloboke.collect.Equivalence;
import com.koloboke.compile.KolobokeMap;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import com.koloboke.function.ObjCharConsumer;
import com.koloboke.function.ObjCharPredicate;
import com.koloboke.function.ObjCharToCharFunction;
import com.koloboke.function.ToCharFunction;
import java.util.function.Function;
import com.koloboke.function.CharBinaryOperator;
import com.koloboke.collect.CharCollection;
import com.koloboke.collect.set.ObjSet;
import com.koloboke.collect.set.ObjSet;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import java.util.Map;


/**
 * 
 * A {@link Map} specialization with {@code Object} keys and {@code char} values.
 * 
 *
 * @see ObjCharMapFactory
 * @see KolobokeMap @KolobokeMap
 */
public interface ObjCharMap<K> extends Map<K, Character>, Container {

    /**
     * Returns the equivalence strategy for keys in this map. All methods in the {@link Map}
     * interface which defined in terms of {@link Object#equals(Object)} equality of key objects
     * (almost all methods, actually), are supposed to use this equivalence instead.
     *
     * @return the equivalence strategy for keys in this map
     * @see com.koloboke.collect.map.hash.HashObjCharMapFactory#withKeyEquivalence
     */
    @Nonnull
    Equivalence<K> keyEquivalence();


    /**
     * Returns the default value of this map, which is used instead of {@code null}
     * in primitive specialization methods, when the key is absent in the map.
     *
     * @return the default value of this map
     * @see ObjCharMapFactory#withDefaultValue(char)
     */
    char defaultValue();



    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #containsValue(char)} instead
     */
    @Override
    @Deprecated
    boolean containsValue(Object value);

    /**
     * Returns {@code true} if this map maps one or more keys to the specified value. This operation
     * will probably require time linear in the map size for most implementations
     * of the {@code ObjCharMap} interface.
     *
     * @param value the {@code char} value whose presence in this map is to be tested
     * @return {@code true} if this map maps one or more keys to the specified value
     */
    boolean containsValue(char value);

    

    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #getChar(Object)} instead
     */
    @Nullable
    @Override
    @Deprecated
    Character get(Object key);

    

    /**
     * Returns the value to which the specified key is mapped, or {@linkplain #defaultValue() default
    value} if this map
     * contains no mapping for the key.
     *
     * 
     * <p>More formally, if this map contains a mapping from a key {@code k} to a value {@code v}
     * such that {@code keyEquivalence() == null ? (key==null ? k==null : key.equals(k)) :
     * keyEquivalence().nullableEquivalent(k, key)}, then this method returns {@code v}; otherwise
     * it returns {@linkplain #defaultValue() default
    value}. (There can be at most one such mapping.)
     *
     * 
     *
     * @param key the key whose associated value is to be returned
     * @return the value to which the specified key is mapped, or {@linkplain #defaultValue() default
    value} if this map
     *         contains no mapping for the key
     * @throws ClassCastException if the key is of an inappropriate type for
     *         this map (optional restriction)
     * @throws NullPointerException if the specified key is null and this map does not permit
     *         null keys (optional restriction)
     */
    
    char getChar(Object key);


    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #getOrDefault(Object, char)} instead
     */
    @Override
    @Deprecated
    Character getOrDefault(Object key, Character defaultValue);

    /**
     * Returns the value to which the specified key is mapped, or {@code defaultValue} if this map
     * contains no mapping for the key.
     *
     * @param key the key whose associated value is to be returned
     * @param defaultValue the value to return if the specified {@code key} is absent in the map
     * @return the value to which the specified key is mapped, or
     *         {@code defaultValue} if this map contains no mapping for the key
     * @throws ClassCastException if the key is of an inappropriate type for
     *         this map (optional restriction)
     * @throws NullPointerException if the specified key is null and this map
     *         does not permit null keys (optional restriction)
     */
    char getOrDefault(Object key, char defaultValue);


    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #forEach(ObjCharConsumer)} instead
     */
    @Override
    @Deprecated
    void forEach(@Nonnull BiConsumer<? super K, ? super Character> action);

    /**
     * Performs the given {@code action} on each entry in this map until all entries
     * have been processed or the action throws an {@code Exception}.
     * Exceptions thrown by the action are relayed to the caller. The entries
     * will be processed in the same order as the entry set iterator unless that
     * order is unspecified in which case implementations may use an order which
     * differs from the entry set iterator.
     *
     * @param action The action to be performed for each entry
     * @see <a href="{@docRoot}/overview-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    void forEach(@Nonnull ObjCharConsumer<? super K> action);


    /**
     * Checks the given {@code predicate} on each entry in this map until all entries
     * have been processed or the predicate returns {@code false} for some entry,
     * or throws an {@code Exception}. Exceptions thrown by the predicate are relayed to the caller.
     *
     * <p>The entries will be processed in the same order as the entry set iterator unless that
     * order is unspecified in which case implementations may use an order which differs from
     * the entry set iterator.
     *
     * <p>If the map is empty, this method returns {@code true} immediately.
     *
     * @return {@code true} if the predicate returned {@code true} for all entries of the map,
     *         {@code false} if it returned {@code false} for the entry
     * @param predicate the predicate to be checked for each entry
     * @see <a href="{@docRoot}/overview-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    boolean forEachWhile(@Nonnull ObjCharPredicate<? super K> predicate);

    /**
     * Returns a new cursor over the entries of this map. It's order is always correspond to the
     * entry set iterator order.
     *
     * <p>Basic cursor usage idiom is: <pre>{@code
     * for (ObjCharCursor<K> cur = map.cursor(); cur.moveNext();) {
     *     // Work with cur.key() and cur.value()
     *     // Call cur.remove() to remove the current entry
     * }}</pre>
     *
     * @return a new cursor over the entries of this map
     * @see <a href="{@docRoot}/overview-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    @Nonnull
    ObjCharCursor<K> cursor();


    @Override
    @Nonnull
    ObjSet<K> keySet();

    @Override
    @Nonnull
    CharCollection values();

    @Override
    @Nonnull
    ObjSet<Entry<K, Character>> entrySet();

    

    

    

    

    

    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #put(Object, char)} instead
     */
    @Override
    @Deprecated
    Character put(K key, Character value);

    /**
     * Associates the specified value with the specified key in this map (optional operation).
     * If the map previously contained a mapping for the key, the old value is replaced
     * by the specified value. (A map {@code m} is said to contain a mapping for a key {@code k}
     * if and only if {@link #containsKey(Object) m.containsKey(k)} would return {@code true}.)
     *
     * @param key key with which the specified value is to be associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with {@code key}, or {@linkplain #defaultValue() default
    value} if there was
     *         no mapping for {@code key}. (A {@linkplain #defaultValue() default
    value} return can also indicate that the map
     *         previously associated {@linkplain #defaultValue() default
    value} with {@code key}.)
     * @throws UnsupportedOperationException if the {@code put} operation
    *         is not supported by this map
     * @throws ClassCastException if the class of the specified key
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified key is null,
     *         and this map does not permit null keys
     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     */
    char put(K key, char value);

    

    

    /**
     * If the specified key is not already associated with a value, associates
     * it with the given value and returns {@code null}, else returns the current value.
     *
     * @param key key with which the specified value is to be associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key, or
     *         {@code null} if there was no mapping for the key.
     *         (A {@code null} return can also indicate that the map
     *         previously associated {@code null} with the key,
     *         if the implementation supports null values.)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     * @throws ClassCastException if the key or value is of an inappropriate type for this map
     * @throws NullPointerException if the specified value is null
     * @throws IllegalArgumentException if some property of the specified key
     *         or value prevents it from being stored in this map
     * @deprecated Use specialization {@link #putIfAbsent(Object, char)} instead
     */
    @Override
    @Nullable
    @Deprecated
    Character putIfAbsent(K key, Character value);

    /**
     * If the specified key is not already associated with a value, associates
     * it with the given value and returns {@linkplain #defaultValue() default
    value}, else returns the current value.
     *
     * @param key key with which the specified value is to be associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key, or {@linkplain #defaultValue() default
    value}
     *         if there was no mapping for the key. (A {@linkplain #defaultValue() default
    value} return
     *         can also indicate that the map previously associated {@linkplain #defaultValue() default
    value}
     *         with the key, if the implementation supports such values.)
     * @throws UnsupportedOperationException if the {@code put} operation
    *         is not supported by this map
     * @throws ClassCastException if the class of the specified key
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified key is null,
     *         and this map does not permit null keys
     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     */
    
    char putIfAbsent(K key, char value);


    

    

    

    

    /**
     * {@inheritDoc}
     * @deprecated Use specialization
     *             {@link #compute(Object, ObjCharToCharFunction)} instead
     */
    @Override
    @Deprecated
    Character compute(K key,
            @Nonnull BiFunction<? super K, ? super Character, ? extends Character> remappingFunction
    );

    /**
     * Attempts to compute a mapping for the specified key and its current mapped value
     * (or {@linkplain #defaultValue() default
    value} if there is no current mapping).
     *
     * <p>If the function itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param remappingFunction the function to compute a value
     * @return the new value associated with the specified key
     *         or if the specified key is null and this map does not support null keys
     * @throws ClassCastException if the class of the
     *         specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    char compute(K key, @Nonnull ObjCharToCharFunction<? super K> remappingFunction);


    /**
     * {@inheritDoc}
     * @deprecated Use specialization
     *             {@link #computeIfAbsent(Object, ToCharFunction)} instead
     */
    @Override
    @Deprecated
    Character computeIfAbsent(K key,
            @Nonnull Function<? super K, ? extends Character> mappingFunction);

    /**
     * If the specified key is not already associated with a value, attempts
     * to compute its value using the given mapping function and enters it into this map
     * . The most common usage is to construct
     * a new object serving as an initial mapped value or memoized result.
     *
     * <p>If the function itself throws an (unchecked) exception, the exception is rethrown,
     * and no mapping is recorded.
     *
     * @param key key with which the specified value is to be associated
     * @param mappingFunction the function to compute a value
     * @return the current (existing or computed) value associated with
     *         the specified key
     *         or if the specified key is null and this map does not support null keys
     * @throws ClassCastException if the class of the
     *         specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    char computeIfAbsent(K key, @Nonnull ToCharFunction<? super K> mappingFunction);


    /**
     * {@inheritDoc}
     * @deprecated Use specialization
     *             {@link #computeIfPresent(Object, ObjCharToCharFunction)} instead
     */
    @Override
    @Deprecated
    Character computeIfPresent(K key,
            @Nonnull BiFunction<? super K, ? super Character, ? extends Character> remappingFunction
    );

    /**
     * If the value for the specified key is present,
     * attempts to compute a new mapping given the key and its current mapped value.
     *
     * <p>If the function itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param remappingFunction the function to compute a value
     * @return the new value associated with the specified key,
     *         or "no entry" value
     *         or if the specified key is null and this map does not support null keys
     * @throws ClassCastException if the class of the
     *         specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    char computeIfPresent(K key, @Nonnull ObjCharToCharFunction<? super K> remappingFunction);


    /**
     * {@inheritDoc}
     * @deprecated Use specialization
     *             {@link #merge(Object, char, CharBinaryOperator)} instead
     */
    @Override
    @Deprecated
    Character merge(K key, Character value,
            @Nonnull BiFunction<? super Character, ? super Character, ? extends Character> remappingFunction);

    /**
     * If the specified key is not already associated with a value, associates
     * it with the given value, otherwise, replaces the value with the results of the given
     * remapping function.
     *
     * This method may be of use when combining multiple mapped values for a key.
     *
     * <p>If the remappingFunction itself throws an (unchecked) exception,
     * the exception is rethrown, and the current mapping is left unchanged.
     *
     * @param key key with which the specified value is to be associated
     * @param value the value to use if absent
     * @param remappingFunction the function to recompute a value if present
     * @return the new value associated with the specified key
     * @throws NullPointerException if the {@code remappingFunction} is null
     *         or if the specified key is null and this map does not support null keys
     *         or if the specified value is null and this map does not support null values
     * @throws ClassCastException if the class of the
     *         specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    char merge(K key, char value, @Nonnull CharBinaryOperator remappingFunction);


    /**
     * Adds the given value {@code addition} to the value associated with the specified key,
     * or to the {@linkplain #defaultValue() default value} if this map contains no mapping for
     * the key, and associates the resulting value with the key.
     *
     * @param key the key to which value add the given value
     * @param addition the value addition
     * @return the new value associated with the specified key
     *         or if the specified key is null and this map does not support null keys
     * @throws ClassCastException if the class of the
     *         specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    char addValue(K key, char addition);

    /**
     * Adds the given value {@code addition} to the value associated with the specified key,
     * or the given {@code initialValue} if this map contains no mapping for the key, and associates
     * the resulting value with the key.
     *
     * <p>This version of {@link #addValue(Object, char)} method is useful if you want
     * to accumulate values from the different initial value, than the {@linkplain #defaultValue()
     * default value} of this map.
     *
     * @param key the key to which value add the given value
     * @param addition the value addition
     * @param initialValue the value to add the given value {@code addition} to, if the map contains
     * no mapping for the given key
     * @return the new value associated with the specified key
     *         or if the specified key is null and this map does not support null keys
     * @throws ClassCastException if the class of the
     *         specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    char addValue(K key, char addition, char initialValue);

    /**
     * Replaces the entry for the specified key only if it is currently mapped to some value.
     *
     * @param key key with which the specified value is associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key, or {@code null} if there was
     *         no mapping for the key. 
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     * @throws NullPointerException if the specified value is null
     * @throws IllegalArgumentException if some property of the specified value
     *         prevents it from being stored in this map
     * @deprecated Use specialization {@link #replace(Object, char)} instead
     */
    @Override
    @Nullable
    @Deprecated
    Character replace(K key, Character value);

    /**
     * Replaces the entry for the specified key only if it is currently mapped to some value.
     *
     * @param key key with which the specified value is associated
     * @param value value to be associated with the specified key
     * @return the previous value associated with the specified key,
     *         or {@linkplain #defaultValue() default
    value} if there was no mapping for the key.
     *         (A {@linkplain #defaultValue() default
    value} return can also indicate that the map
     *         previously associated {@linkplain #defaultValue() default
    value} with the key,
     *         if the implementation supports such values.)
     * @throws ClassCastException if the class of the specified key
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified key is null,
     *         and this map does not permit null keys
     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    
    char replace(K key, char value);


    /**
     * Replaces the entry for the specified key only if currently mapped to the specified value.
     *
     * @param key key with which the specified value is associated
     * @param oldValue value expected to be associated with the specified key
     * @param newValue value to be associated with the specified key
     * @return {@code true} if the value was replaced
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     * @throws NullPointerException if the specified value is null
     * @throws IllegalArgumentException if some property of the specified value
     *         prevents it from being stored in this map
     * @deprecated Use specialization
     *             {@link #replace(Object, char, char)} instead
     */
    @Override
    @Deprecated
    boolean replace(K key, Character oldValue, Character newValue);

    /**
     * Replaces the entry for the specified key only if currently mapped to the specified value.
     *
     * @param key key with which the specified value is associated
     * @param oldValue value expected to be associated with the specified key
     * @param newValue value to be associated with the specified key
     * @return {@code true} if the value was replaced
     * @throws ClassCastException if the class of the specified key
     *         prevents it from being stored in this map
     * @throws NullPointerException if the specified key is null,
     *         and this map does not permit null keys
     * @throws IllegalArgumentException if some property of a specified key
     *         or value prevents it from being stored in this map
     * @throws UnsupportedOperationException if the {@code put} operation
     *         is not supported by this map
     */
    boolean replace(K key, char oldValue, char newValue);


    /**
     * {@inheritDoc}
     * @deprecated Use specialization {@link #replaceAll(ObjCharToCharFunction)} instead
     */
    @Override
    @Deprecated
    void replaceAll(
            @Nonnull BiFunction<? super K, ? super Character, ? extends Character> function);

    /**
     * Replaces each entry's value with the result of invoking the given function on that entry,
     * in the order entries are returned by an entry set iterator, until all entries have been
     * processed or the function throws an exception.
     *
     * @param function the function to apply to each entry
     * @throws UnsupportedOperationException if the {@code set} operation
     *         is not supported by this map's entry set iterator
     * @throws IllegalArgumentException if some property of a replacement value
     *         prevents it from being stored in this map (optional restriction)
     */
    void replaceAll(@Nonnull ObjCharToCharFunction<? super K> function);


    

    /**
     * {@inheritDoc}
     * @deprecated Use specialization
     *             {@link #removeAsChar(Object)} instead
     */
    @Override
    @Nullable
    @Deprecated
    Character remove(Object key);

    /**
     * Removes the mapping for a key from this map if it is present (optional operation).
     *
     * <p>More formally, if this map contains a mapping from a key {@code k} to a value {@code v}
     * such that {@code keyEquivalence() == null ? (key==null ? k==null : key.equals(k)) :
     * keyEquivalence().nullableEquivalent(k, key)}, that mapping is removed.
     * (The map can contain at most one such mapping.)
     *
     * <p>Returns the value to which this map previously associated the key, or {@linkplain #defaultValue() default
    value}
     * if the map contained no mapping for the key.
     *
     * <p>A return value of {@linkplain #defaultValue() default
    value} does not <i>necessarily</i> indicate that the map
     * contained no mapping for the key; it's also possible that the map
     * explicitly mapped the key to {@linkplain #defaultValue() default
    value}.
     *
     * <p>The map will not contain a mapping for the specified key once the
     * call returns.
     *
     * @param key key whose mapping is to be removed from the map
     * @return the previous value associated with {@code key}, or {@linkplain #defaultValue() default
    value} if there was
     *         no mapping for {@code key}
     * @throws ClassCastException if the class of the specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws NullPointerException if the specified key is null,
     *         and this map does not permit null keys (optional restriction)
     * @throws UnsupportedOperationException if the {@code remove} operation
     *         is not supported by this map
     */
    
    char removeAsChar(Object key);


    /**
     * Removes the entry for the specified key only if it is currently mapped to the specified
     * value.
     *
     * @param key key with which the specified value is associated
     * @param value value expected to be associated with the specified key
     * @return {@code true} if the value was removed
     * @throws NullPointerException if the specified value is null
               , or if the specified key is null,
     *         and this map does not permit null keys (optional restriction)
     * @throws ClassCastException if the class of the specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws UnsupportedOperationException if the {@code remove} operation
     *         is not supported by this map
     * @deprecated Use specialization {@link #remove(Object, char)} instead
     */
    @Override
    @Deprecated
    boolean remove(Object key, Object value);

    /**
     * Removes the entry for the specified key only if it is currently mapped to the specified
     * value.
     *
     * @param key key with which the specified value is associated
     * @param value value expected to be associated with the specified key
     * @return {@code true} if the value was removed
     * @throws ClassCastException if the class of the specified key
     *         prevents it from being stored in this map (optional restriction)
     * @throws NullPointerException if the specified key is null,
     *         and this map does not permit null keys (optional restriction)
     * @throws UnsupportedOperationException if the {@code remove} operation
     *         is not supported by this map
     */
    boolean remove(Object key, char value);

    /**
     * Removes all of the entries of this collection that satisfy the given predicate.
     * Errors or runtime exceptions thrown during iteration or by the predicate are relayed
     * to the caller.
     *
     * @param filter a predicate which returns {@code true} for elements to be removed
     * @return {@code true} if any elements were removed
     * @throws NullPointerException if the specified filter is null
     * @throws UnsupportedOperationException if elements cannot be removed from this collection.
     *         Implementations may throw this exception if a matching element cannot be removed
     *         or if, in general, removal is not supported.
     * @see <a href="{@docRoot}/overview-summary.html#iteration">
     *     Comparison of iteration options in the library</a>
     */
    boolean removeIf(@Nonnull ObjCharPredicate<? super K> filter);
}
