/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.mapping.rule.instructions;

import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.set.hash.TIntHashSet;

import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.utils.triggers.IModification;
import org.simantics.mapping.constraint.instructions.IInstruction;

public class IfRuleInstruction implements IRuleInstruction {
	IInstruction condition;
	IRuleInstruction rule;
	IRuleInstruction elseRule;	
	
	public IfRuleInstruction(IInstruction condition, IRuleInstruction rule, IRuleInstruction elseRule) {
		this.condition = condition;
		this.rule = rule;
		this.elseRule = elseRule;
	}
	
	public IfRuleInstruction(IInstruction condition, IRuleInstruction rule) {
		this(condition, rule, null);
	}	
	
	@Override
	public IModification execute(ReadGraph g, final Object[] bindings) throws DatabaseException {
		if(DEBUG)
			System.out.println("IfRuleInstruction: condition");
		Object cont = condition.query(g, bindings);
		if(cont == IInstruction.FAILURE) {
			if(DEBUG)
			  System.out.println("IfRuleInstruction: else");
			if(elseRule!=null)
				return elseRule.execute(g, bindings);
			else
				return null;
		}
		do {
			if(DEBUG)
				System.out.println("IfRuleInstruction: modification");
			final IModification modi = rule.execute(g, bindings);
			if(modi != null) {
				if(cont==null)
					return modi;
				final Object curCont = condition.next(g, bindings, cont);
				if(curCont==IInstruction.FAILURE)
					return modi;
				return new IModification() {
					@Override
					public void perform(WriteGraph g) throws DatabaseException {
						modi.perform(g);
						Object cont = curCont;
						do {
							rule.doExecute(g, bindings);
							if(cont==null)
								break;
							cont = condition.next(g, bindings, cont);
						} while(cont != IInstruction.FAILURE);
					}				
				};	
			}
			if(cont==null)
				break;
			cont = condition.next(g, bindings, cont);
		} while(cont != IInstruction.FAILURE);
		return null;
	}
	
	@Override
	public void doExecute(WriteGraph g, Object[] bindings) throws DatabaseException {
		Object cont = condition.query(g, bindings);
		if(cont == IInstruction.FAILURE) {
			if(elseRule!=null) 
				elseRule.doExecute(g, bindings);
			return;
		}
		do {
			rule.doExecute(g, bindings);
			if(cont==null)
				break;
			cont = condition.next(g, bindings, cont);
		} while(cont != IInstruction.FAILURE);
	}
	
	@Override
	public void collectVariables(TIntHashSet reads, TIntHashSet writes) {
		condition.collectVariables(reads, writes);
		rule.collectVariables(reads, writes);
		if(elseRule != null)
			elseRule.collectVariables(reads, writes);
	}

	@Override
	public void mapVariables(TIntIntHashMap map) {
		condition.mapVariables(map);
		rule.mapVariables(map);
		if(elseRule != null)
			elseRule.mapVariables(map);
	}

	@Override
	public void toString(StringBuilder b, int indent) {		
		b.append("if   ");
		condition.toString(b, indent+1);
		
		b.append('\n');		
		for(int i=0;i<indent;++i)
			b.append(INDENTATION);
		b.append("then ");
		rule.toString(b, indent+1);		
		
		if(elseRule != null) {
			b.append('\n');
			for(int i=0;i<indent;++i)
				b.append(INDENTATION);
			b.append("else ");
			elseRule.toString(b, indent+1);
		}
	}
}
