/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.history.util;

import java.util.Iterator;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.ArrayType;

/**
 * This class binds median class to as an array(double) where every second value
 * represents value and odd weight.
 */
public class WeightedMedianBinding extends ArrayBinding {

	public WeightedMedianBinding() {
		super(new ArrayType(Bindings.DOUBLE.type()), Bindings.DOUBLE);
	}
	
	@Override
	public Object create() {
		return new WeightedMedian();
	}

	@Override
	public Object create(int length, Iterator<Object> it) throws BindingException {		
		WeightedMedian result = new WeightedMedian( length );
		int count = length/2;
		WeightedMedian.Item[] items = new WeightedMedian.Item[count];		
		int index = 0;
		double value=0.0, weight=0.0;
		while (it.hasNext()) {
			double d = (Double) it.next();
			if ( index%2==0 ) {
				value = d;
			} else {
				weight = d;
				WeightedMedian.Item i = new WeightedMedian.Item(weight, value);
				items[ index/2 ] = i;
			}
			index++;
		}
		if ( count>0 ) {
			items[0].next = items[1];
			items[count-1].prev = items[count-2];
		}		
		for (index=1; index<count-1; index++) {
			items[index].next = items[index+1];
			items[index].prev = items[index-1];
		}
		result.median = items[count/2];
		result.itemCount = count;
		return result;
	}

	@Override
	public Object create(Object[] array) throws BindingException {
		int count = array.length/2;
		WeightedMedian result = new WeightedMedian();
		WeightedMedian.Item[] items = new WeightedMedian.Item[count];		
		double value=0.0, weight=0.0;
		for (int index=0; index<array.length; index++) {
			double d = (Double) array[index];
			if ( index%2==0 ) {
				value = d;
			} else {
				weight = d;
				WeightedMedian.Item i = new WeightedMedian.Item(weight, value);
				items[ index/2 ] = i;
			}
			index++;
		}
		if ( count>0 ) {
			items[0].next = items[1];
			items[count-1].prev = items[count-2];
		}		
		for (int index=1; index<count-1; index++) {
			items[index].next = items[index+1];
			items[index].prev = items[index-1];
		}
		result.median = items[count/2];
		result.itemCount = count;
		return result;
	}

	@Override
	public void add(Object array, int index, Object element) throws BindingException, IndexOutOfBoundsException {
		throw new BindingException("not implemented");
	}

	@Override
	public void remove(Object array, int index, int count) throws BindingException {
		throw new BindingException("not implemented");
	}

	@Override
	public Object get(Object array, int index) throws BindingException {
		int pos = index/2;
		WeightedMedian median = (WeightedMedian) array;
		int c = median.itemCount*2;
		if ( index<0 || index>=c ) throw new BindingException("Index "+index+" out of bounds "+c);
		WeightedMedian.Item i = median.head();
		for (int ix=1; ix<pos; ix++) i = i.next;
		return index%2==0?i.value:i.weight;
	}

	@Override
	public void getAll(Object array, Object[] result) throws BindingException {
		int c = size( array );
		if (result.length<c) throw new BindingException("Array too small");
		WeightedMedian median = (WeightedMedian) array;
		int ix = 0;
		WeightedMedian.Item i = median.head();
		while (i != null) {
			result[ix] = i.value; ix++;
			result[ix] = i.weight; ix++;
			i = i.next;
		}
	}

	@Override
	public void set(Object array, int index, Object value) throws BindingException {
		remove(array, index);
		add(array, value);
	}

	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		int size = size(array);
		if (size<newSize) newSize = size;
		remove( array, size-newSize, newSize);
	}

	@Override
	public int size(Object array) throws BindingException {
		WeightedMedian median = (WeightedMedian) array;
		return median.itemCount*2;
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof WeightedMedian;
	}	

	@Override
	public boolean isResizable() {
		return true;
	}
	
}


