package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.List;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer.NonRecursiveSerializer;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.util.Range;
import org.simantics.databoard.util.binary.Endian;

public class DoubleArraySerializer extends NonRecursiveSerializer {

	Range length;
	Integer fixedLength, fixedSize;
	
	public DoubleArraySerializer(ArrayBinding binding)
	{
		ArrayType arrayType = (ArrayType) binding.type();			
		this.length = arrayType.getLength();
		if (length!=null && length.getLower().equals(length.getUpper()) && length.getLower().getValue()!=null)
		{
			fixedLength = length.getLower().getValue().intValue();
			fixedSize = fixedLength * 8;
		}
	}

	@Override
	public Object deserialize(byte[] data) throws IOException {
		int length = fixedSize != null ? fixedLength : Endian.readInt(data);
		if (length<0) throw new SerializationException("Cannot use negative array length");
		if (data.length < length*8+4)
			throw new SerializationException("Malformed data. Deserialization aborted. (Wrong binding?)");

		double[] array = new double[length];
		for(int i=0,offset=4;i<array.length;++i, offset += 8)
			array[i] = Double.longBitsToDouble(Endian.readLong(data, offset));
		return array;
	}

	@Override
	public void deserialize(byte[] data, Object obj) throws IOException {
		int length = fixedLength != null ? fixedLength : Endian.readInt(data);
		double[] array = (double[]) obj;
		if (length!=array.length) throw new SerializationException("primitive array is size immutable");
		if (data.length < length*8+4)
			throw new SerializationException("Malformed data. Deserialization aborted. (Wrong binding?)");
		for (int i=0, offset = 4; i<array.length;i++, offset += 8)
			array[i] = Double.longBitsToDouble(Endian.readLong(data, offset));
	}

	@Override
	public Object deserialize(DataInput in)
	throws IOException 
	{
		int length = fixedSize != null ? fixedLength : in.readInt();
		if (length<0) throw new SerializationException("Cannot use negative array length");
		assertRemainingBytes(in, length*8L);					
		
		double[] array = new double[length];
		for(int i=0;i<array.length;++i)
			array[i] = in.readDouble();
		return array;
	}
	
	public Object deserializeToTry(DataInput in, List<Object> identities, Object obj) throws IOException
	{
		int length = fixedLength != null ? fixedLength : in.readInt();
		double[] array = (double[]) obj;
		if (length!=array.length) array = new double[ length ];
		assertRemainingBytes(in, length*8L);					
		
		for (int i=0; i<array.length;i++)
			array[i] = in.readDouble();
		
		return array;
	}

	@Override
	public void deserializeTo(DataInput in, Object obj) throws IOException {
		int length = fixedLength != null ? fixedLength : in.readInt();		
		double[] array = (double[]) obj;
		if (length!=array.length) throw new SerializationException("primitive array is size immutable");
		assertRemainingBytes(in, length*8L);
		
		for (int i=0; i<array.length;i++)
			array[i] = in.readDouble();
	}
	
	@Override
	public void skip(DataInput in)
	throws IOException {
		int length = fixedSize != null ? fixedLength : in.readInt();			
		in.skipBytes(length * 8);
	}
	
	@Override
	public void serialize(DataOutput out, Object obj)
	throws IOException 
	{
		double[] array = (double[])obj;
		if (fixedSize==null) 
			out.writeInt(array.length);
		for(double f : array)
			out.writeDouble(f);
	}

	@Override
	public Integer getConstantSize() {
		return fixedSize;
	}

	@Override
	public int getSize(Object obj) {
		if (fixedSize!=null) return fixedSize;
		double[] array = (double[])obj;			
		return 4 + 8 * array.length;
	}
	
	@Override
	public int getMinSize() {
		return fixedSize != null ? fixedSize : 4;
	}
	
}

