/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.file;

import java.io.File;

import org.simantics.databoard.accessor.Accessor;
import org.simantics.databoard.accessor.CloseableAccessor;
import org.simantics.databoard.accessor.binary.BinaryObject;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.binding.Binding;

/**
 * File accessor is an interface for data that is located in a file in byte format.
 * There are two ways of writing to the files. 
 *   A) Using the writing methods of Accessor+subtypes, eg. setValue, putValue, ...
 *      They guarantee the value is flushed to the file upon return
 *   B) Using the write methods of FileAccessor+subtypes, eg. setValueNoflush, putValueNoflush ...
 *      They do not guarantee flush upon return. Explicit flush() call is required.
 *      The value resides in memory cache and are available for reading event before flushing.
 *
 * see FileAccessorExample
 * @see BinaryObject File/Memory Byte[] implementation
 * @see FileArrayAccessor
 * @see FileBooleanAccessor
 * @see FileByteAccessor
 * @see FileDoubleAccessor
 * @see FileFloatAccessor
 * @see FileIntegerAccessor
 * @see FileLongAccessor
 * @see FileMapAccessor
 * @see FileOptionalAccessor
 * @see FileRecordAccessor
 * @see FileStringAccessor
 * @see FileUnionAccessor
 * @see FileVariantAccessor
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public interface FileAccessor extends Accessor, CloseableAccessor {

	/**
	 * Get the file
	 * 
	 * @return file
	 */
	File file();
	
	/**
	 * Close the file. It is allowed to call this method more than once.
	 * 
	 * @throws AccessorException
	 */
	void close() throws AccessorException;
	
	/**
	 * Flush the internal write buffer to the disc. This is needed if 
	 * setXXNoFlush() methods are used.
	 * <p>
	 * 
	 * Note, all the write methods of {@link Accessor} and its sub-interfaces 
	 * guarantee Durability. They flush the value immediately and do not require
	 * separate {@link #flush()}.
	 * 
	 * @throws AccessorException
	 */
	void flush() throws AccessorException;
	
	/**
	 * Reset internal buffer. If there unwritten changes, they are flushed.
	 * 
	 * @throws AccessorException
	 */
	void reset() throws AccessorException;
	
	/**
	 * Write a value to the file without flushing the writebuffer yet.
	 * The write doesn't become durable until the value is flushed 
	 * with {@link #flush()}.
	 * 
	 * @param binding
	 * @param newValue
	 * @throws AccessorException
	 */
	void setValueNoflush(Binding binding, Object newValue) throws AccessorException;
	
	
	
}

