/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;
import org.simantics.ui.internal.Activator;
import org.simantics.utils.strings.StringUtils;

/**
 * @author Tuukka Lehtonen
 */
public class DoubleClickExtensionManager implements IExtensionChangeHandler {

    private final static String                NAMESPACE    = Activator.PLUGIN_ID;

    private final static String                EP_NAME      = "doubleClick";

    private ExtensionTracker                   tracker;

    private IDoubleClickExtension[]            extensions   = new IDoubleClickExtension[0];

    private static DoubleClickExtensionManager INSTANCE;


    public static synchronized DoubleClickExtensionManager getInstance() {
        if (INSTANCE == null)
            INSTANCE = new DoubleClickExtensionManager();
        return INSTANCE;
    }

    public static synchronized void dispose() {
        if (INSTANCE != null) {
            INSTANCE.close();
            INSTANCE = null;
        }
    }


    private DoubleClickExtensionManager() {
        tracker = new ExtensionTracker();

        // Cache defined actions
        IExtensionPoint expt = Platform.getExtensionRegistry().getExtensionPoint(NAMESPACE, EP_NAME);
        loadExtensions(expt.getConfigurationElements());

        // Start tracking for new and removed extensions
        IFilter filter = ExtensionTracker.createExtensionPointFilter(expt);
        tracker.registerHandler(this, filter);
    }

    private void close() {
        tracker.close();
        tracker = null;
        extensions = new IDoubleClickExtension[0];
    }

    public IDoubleClickExtension[] getExtensions() {
        return extensions;
    }

    private void loadExtensions(IConfigurationElement[] elements) {
        Set<IDoubleClickExtension> newExtensions = new HashSet<IDoubleClickExtension>(Arrays.asList(extensions));

        for (IConfigurationElement el : elements) {
            String name = StringUtils.safeString(el.getAttribute("name"));
            double priority = 0;
            try {
                String p = el.getAttribute("priority");
                if (p != null) {
                    priority = Double.parseDouble(p);
                }
            } catch (NumberFormatException e) {
            }
            IDoubleClickExtension ext = new DoubleClickExtension(el, name, priority);

            // Start tracking the new extension object, its removal will be notified of
            // with removeExtension(extension, Object[]).
            tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

            newExtensions.add(ext);
        }

        // Sort in descending priority order
        List<IDoubleClickExtension> unsorted = new ArrayList<IDoubleClickExtension>(newExtensions);
        Collections.sort(unsorted, new Comparator<IDoubleClickExtension>() {
            @Override
            public int compare(IDoubleClickExtension o1, IDoubleClickExtension o2) {
                return Double.compare(o2.getPriority(), o1.getPriority());
            }
        });

        // Atomic assignment
        this.extensions = unsorted.toArray(new IDoubleClickExtension[unsorted.size()]);
    }

    @Override
    public void addExtension(IExtensionTracker tracker, IExtension extension) {
        loadExtensions(extension.getConfigurationElements());
    }

    @Override
    public void removeExtension(IExtension extension, Object[] objects) {
        Set<IDoubleClickExtension> newExtensions = new HashSet<IDoubleClickExtension>(Arrays.asList(extensions));

        for (Object o : objects) {
            tracker.unregisterObject(extension, o);
            newExtensions.remove(o);
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new IDoubleClickExtension[newExtensions.size()]);
    }

}
