/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import java.io.FileNotFoundException;
import java.io.IOException;

import junit.framework.TestCase;

import org.simantics.databoard.util.binary.BinaryMemory;
import org.simantics.databoard.util.binary.BinaryReadable;
import org.simantics.databoard.util.binary.BinaryWriteable;
import org.simantics.databoard.util.binary.RandomAccessBinary;
import org.simantics.databoard.util.binary.RandomAccessBinary.ByteSide;

public class TestBinaryMemory extends TestCase {

	public static final int ITERATIONS = 50;
	RandomAccessBinary file;
	
	public void setUp() throws Exception {
		file = new BinaryMemory(4096);
	}
	
	public void tearDown() throws Exception {
		file.close();
	}
	
	public void testInsertRemove()
	throws IOException
	{
		// Write 4 MB file
		for (int i=0; i<1024*1024; i++)
		{
			file.writeInt(i ^0x53);
		}
		file.flush();

		int iterCount = 16;
		long totInsertTime = 0;
		long totRemoveTime = 0;
		
		for (int iter=0; iter<iterCount; iter++) {
		// Verify content
		// 1. half
		file.position(0L);
		for (int i=0; i<1024*1024; i++)
			assertEquals(i^0x53, file.readInt());
		
		// Add 256 kb at 2MB
		System.gc();
		long startTime = System.currentTimeMillis();
		file.position(512*1024*4L);
		file.insertBytes(256*1024, ByteSide.Left);
		long elapsedTime = System.currentTimeMillis() - startTime;
		totInsertTime += elapsedTime;
		System.out.println("Insert 256kb in front of 2MB, time: "+elapsedTime);
		
		// Verify file size
		assertEquals(4*1024*1024 + 256*1024, file.length()); 
		
		// Verify content
		// 1. half
		file.position(0);
		for (int i=0; i<512*1024; i++)
			assertEquals(i^0x53, file.readInt());
		// 2. half
		file.position(512*1024*4+256*1024);
		for (int i=512*1024; i<1024*1024; i++)
			assertEquals(i^0x53, file.readInt());

		// Remove content
		System.gc();
		startTime = System.currentTimeMillis();
		file.position(512*1024*4);
		file.removeBytes(256*1024, ByteSide.Left);
		elapsedTime = System.currentTimeMillis() - startTime;
		totRemoveTime += elapsedTime;
		System.out.println("remove 256kb in front of 2,25MB, time: "+elapsedTime);
		
		// Verify file size
		assertEquals(4*1024*1024, file.length());
		
		// Verify content
		// 1. half
		file.position(0);
		for (int i=0; i<1024*1024; i++)
			assertEquals(i^0x53, file.readInt());
		}

		System.out.println("Average insert time: " + totInsertTime / iterCount);
		System.out.println("Average remove time: " + totRemoveTime / iterCount);
	}
	
	public void testWriteRead() 
	throws FileNotFoundException, IOException 
	{		
		long time4 = measureWrite(file);
		System.out.println();
		System.out.println("Write Time : "+time4);

		file.flush();
		file.position(0);
		
		time4 = measureRead(file);				
		System.out.println("Read Time : "+time4);
	}
	
	static long measureWrite(BinaryWriteable write) 
	throws IOException
	{		
		System.gc();
		byte[] data = new byte[1024];
		for (int i=0; i<data.length; i++)
			data[i] = (byte) i;
		
		long startTime = System.currentTimeMillis();		
		for (int iter=0; iter<ITERATIONS; iter++)
		{
			if (iter % 10==0) System.out.print(".");
			for (int i=0; i<256; i++)
				write.write((byte)i);
			for (int i=0; i<256; i++)
				write.writeDouble(i);
			for (int i=0; i<256; i++)
				write.writeFloat(i);
			for (int i=0; i<256; i++)
				write.writeInt(i);
			for (int i=0; i<256; i++)
				write.writeShort((short)i);
			write.write(data);			
		}		
		write.flush();
		long elapsedTime = System.currentTimeMillis() - startTime;		
		
		return elapsedTime;
	}
	
	static long measureRead(BinaryReadable read) 
	throws IOException
	{		
		System.gc();
		byte[] data = new byte[1024];
		
		long startTime = System.currentTimeMillis();		
		for (int iter=0; iter<ITERATIONS; iter++)
		{
			if (iter % 10==0) System.out.print(".");
			for (int i=0; i<256; i++) 
				assertEquals((byte) i, read.readByte());
			for (int i=0; i<256; i++)
				assertEquals((double) i, read.readDouble());
			for (int i=0; i<256; i++)
				assertEquals((float) i, read.readFloat());
			for (int i=0; i<256; i++)
				assertEquals((int) i, read.readInt());
			for (int i=0; i<256; i++)
				assertEquals((short) i, read.readShort());
			read.readFully(data);
			for (int i=0; i<data.length; i++)
				assertEquals(data[i], (byte) i);
		}		
		long elapsedTime = System.currentTimeMillis() - startTime;
		
		return elapsedTime;
	}
	
}

