package org.simantics.spreadsheet;

import java.io.Serializable;

import org.simantics.spreadsheet.solver.formula.FormulaError2;

public class SpreadsheetMatrix implements Serializable {

    private static final long serialVersionUID = 5729700370527844640L;

    private int w;
    private int h;
    public Object[] values;

    public SpreadsheetMatrix(int w, int h) {
        this.w = w;
        this.h = h;
        this.values = new Object[w*h];
    }

    public Object get(int row, int column) {
        return values[w*row+column];
    }

    public void set(int row, int column, Object value) {
        values[w*row+column] = value;
    }

    public int getWidth() {
        return w;
    }

    public int getHeight() {
        return h;
    }

    public double sum() {
        double result = 0;
        for(int i=0;i<values.length;i++) {
            Number n = Spreadsheets.asValidNumber(values[i]);
            if(n != null) result += n.doubleValue();
        }
        return result;
    }

    public Object sumWithFormulaError(){
        Double result = 0.0;
        for(int i = 0; i < values.length; i++){
            Object obj = values[i];
            if(obj != null){
                FormulaError2 error = FormulaError2.forObject(obj);
                if(error!=null) return error.getString();

                Number n = Spreadsheets.asValidNumber(obj);
                if(n != null) result += n.doubleValue();
            }
        }
        return result;
    }

    public double average() {
        double result = 0;
        double count = 0;
        for(int i=0;i<values.length;i++) {
            Number n = Spreadsheets.asValidNumber(values[i]);
            if(n != null) {
                result += n.doubleValue();
                count++;
            }
        }
        if(count == 0) return result;
        return result / count;
    }

    public double countOfActualDoubleValues(){
        double count = 0.0;
        for(int i = 0; i < values.length; i++){
            Number n = Spreadsheets.asValidNumber(values[i]);
            if(n!=null) count++;
        }
        return count;
    }

    public Object productWithFormulaError(){
        Double result = 1.0;
        boolean atLeastOne = false;
        for(int i = 0; i < values.length; i++){
            Object obj = values[i];
            if(obj != null){
                FormulaError2 error = FormulaError2.forObject(obj);
                if(error!=null) return error.getString();

                Number n = Spreadsheets.asValidNumber(values[i]);
                if(n != null) {
                    double dval = n.doubleValue();
                    if(dval<=0) return FormulaError2.NUM.getString();
                    result = result*dval;
                    atLeastOne = true;
                }
            }
        }
        if(atLeastOne) return result;
        else return 0.0;
    }

    public SpreadsheetMatrix pow(Object exponent_) {
        if(exponent_ instanceof SpreadsheetMatrix) {
            SpreadsheetMatrix exponent = (SpreadsheetMatrix)exponent_;
            SpreadsheetMatrix result = new SpreadsheetMatrix(exponent.w, h);
            for(int i=0;i<h;i++) {
                for(int j=0;j<exponent.w;j++) {
                    result.set(i,j, Math.pow(Spreadsheets.asNumber(get(i, 0)),
                            Spreadsheets.asNumber(exponent.get(0, j))));
                }
            }
            return result;
        } else {
            double n = Spreadsheets.asNumber(exponent_);
            SpreadsheetMatrix result = new SpreadsheetMatrix(w, h);
            for(int i=0;i<result.values.length;i++) {
                result.values[i] = Math.pow(Spreadsheets.asNumber(values[i]), n);
            }
            return result;
        }
    }


}