/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.koloboke.collect.map.hash;

import com.koloboke.collect.map.CharLongMap;
import com.koloboke.compile.KolobokeMap;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.ServiceLoader;


/**
 * This class consists only of static factory methods to construct {@code HashCharLongMap}s, and
 * the default {@link HashCharLongMapFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashCharLongMap
 * @see KolobokeMap @KolobokeMap
 */
public final class HashCharLongMaps {
    
    private static class DefaultFactoryHolder {
        private static final HashCharLongMapFactory defaultFactory =
                ServiceLoader.load(HashCharLongMapFactory.class).iterator().next();
    }

    /**
     * Returns the default {@link HashCharLongMapFactory} implementation, to which
     * all static methods in this class delegate.
     *
     
     
     * @return the default {@link HashCharLongMapFactory} implementation
     * @throws RuntimeException if no implementations
     *         of {@link HashCharLongMapFactory} are provided
     */
    @Nonnull
    public static  HashCharLongMapFactory getDefaultFactory() {
        return (HashCharLongMapFactory) DefaultFactoryHolder.defaultFactory;
    }

    

    

    

    


    /**
     * Constructs a new empty mutable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap() newMutableMap()}.
     *
     
     
     * @return a new empty mutable map
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap() {
        return getDefaultFactory().newMutableMap();
    }

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(int) newMutableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(int expectedSize) {
        return getDefaultFactory().newMutableMap(expectedSize);
    }

    
    
    
    


    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Map,
     * Map, int) newMutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Map, Map,
     * Map, int) newMutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Map, Map,
     * Map, Map, int) newMutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map, int) newMutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4,
            @Nonnull Map<Character, Long> map5, int expectedSize) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(
     * Consumer, int) newMutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(@Nonnull
            Consumer<com.koloboke.function.CharLongConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(char[], long[], int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull char[] keys, @Nonnull long[] values, int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Character[],
     * Long[], int) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Character[] keys, @Nonnull Long[] values, int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Iterable, Iterable, int
     * ) newMutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<Long> values
            , int expectedSize) {
        return getDefaultFactory().newMutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(
     * Map) newMutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map) {
        return getDefaultFactory().newMutableMap(map);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Map,
     * Map) newMutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2) {
        return getDefaultFactory().newMutableMap(map1, map2);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Map, Map,
     * Map) newMutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3) {
        return getDefaultFactory().newMutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Map, Map,
     * Map, Map) newMutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Map, Map,
     * Map, Map, Map) newMutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4,
            @Nonnull Map<Character, Long> map5) {
        return getDefaultFactory().newMutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(
     * Consumer) newMutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(@Nonnull
            Consumer<com.koloboke.function.CharLongConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newMutableMap(entriesSupplier);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(char[], long[]
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull char[] keys, @Nonnull long[] values) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Character[],
     * Long[]) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Character[] keys, @Nonnull Long[] values) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMap(Iterable, Iterable
     * ) newMutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashCharLongMap newMutableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<Long> values
            ) {
        return getDefaultFactory().newMutableMap(keys, values);
    }

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMapOf(char, long
     * ) newMutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
    public static  HashCharLongMap newMutableMapOf(
            char k1, long v1) {
        return getDefaultFactory().newMutableMapOf(k1, v1);
    }

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMapOf(char, long,
     * char, long) newMutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newMutableMapOf(
            char k1, long v1, char k2, long v2) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMapOf(char, long,
     * char, long, char, long) newMutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newMutableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMapOf(char, long,
     * char, long, char, long, char, long
     * ) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newMutableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3, char k4, long v4) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newMutableMapOf(char, long,
     * char, long, char, long, char, long,
     * char, long) newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newMutableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3, char k4, long v4,
            char k5, long v5) {
        return getDefaultFactory().newMutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    /**
     * Constructs a new empty updatable map of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap() newUpdatableMap()}.
     *
     
     
     * @return a new empty updatable map
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap() {
        return getDefaultFactory().newUpdatableMap();
    }

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(int) newUpdatableMap(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(int expectedSize) {
        return getDefaultFactory().newUpdatableMap(expectedSize);
    }

    
    
    
    


    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Map,
     * Map, int) newUpdatableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Map, Map,
     * Map, int) newUpdatableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, int) newUpdatableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map, int) newUpdatableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4,
            @Nonnull Map<Character, Long> map5, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(
     * Consumer, int) newUpdatableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(@Nonnull
            Consumer<com.koloboke.function.CharLongConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(char[], long[], int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull char[] keys, @Nonnull long[] values, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Character[],
     * Long[], int) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Character[] keys, @Nonnull Long[] values, int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Iterable, Iterable, int
     * ) newUpdatableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<Long> values
            , int expectedSize) {
        return getDefaultFactory().newUpdatableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(
     * Map) newUpdatableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map) {
        return getDefaultFactory().newUpdatableMap(map);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Map,
     * Map) newUpdatableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2) {
        return getDefaultFactory().newUpdatableMap(map1, map2);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Map, Map,
     * Map) newUpdatableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Map, Map,
     * Map, Map) newUpdatableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Map, Map,
     * Map, Map, Map) newUpdatableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4,
            @Nonnull Map<Character, Long> map5) {
        return getDefaultFactory().newUpdatableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(
     * Consumer) newUpdatableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(@Nonnull
            Consumer<com.koloboke.function.CharLongConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newUpdatableMap(entriesSupplier);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(char[], long[]
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull char[] keys, @Nonnull long[] values) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Character[],
     * Long[]) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Character[] keys, @Nonnull Long[] values) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMap(Iterable, Iterable
     * ) newUpdatableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<Long> values
            ) {
        return getDefaultFactory().newUpdatableMap(keys, values);
    }

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMapOf(char, long
     * ) newUpdatableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMapOf(
            char k1, long v1) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1);
    }

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMapOf(char, long,
     * char, long) newUpdatableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMapOf(
            char k1, long v1, char k2, long v2) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMapOf(char, long,
     * char, long, char, long) newUpdatableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMapOf(char, long,
     * char, long, char, long, char, long
     * ) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3, char k4, long v4) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newUpdatableMapOf(char, long,
     * char, long, char, long, char, long,
     * char, long) newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newUpdatableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3, char k4, long v4,
            char k5, long v5) {
        return getDefaultFactory().newUpdatableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }


    
    
    
    


    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Map,
     * Map, int) newImmutableMap(map1, map2, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Map, Map,
     * Map, int) newImmutableMap(map1, map2, map3, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Map, Map,
     * Map, Map, int) newImmutableMap(map1, map2, map3, map4, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, expectedSize);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map, int) newImmutableMap(map1, map2, map3, map4, map5, expectedSize)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4,
            @Nonnull Map<Character, Long> map5, int expectedSize) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, map5, expectedSize);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(
     * Consumer, int) newImmutableMap(entriesSupplier, expectedSize)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(@Nonnull
            Consumer<com.koloboke.function.CharLongConsumer> entriesSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableMap(entriesSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(char[], long[], int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull char[] keys, @Nonnull long[] values, int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Character[],
     * Long[], int) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Character[] keys, @Nonnull Long[] values, int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Iterable, Iterable, int
     * ) newImmutableMap(keys, values, expectedSize)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<Long> values
            , int expectedSize) {
        return getDefaultFactory().newImmutableMap(keys, values, expectedSize);
    }
    
    
    
    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(
     * Map) newImmutableMap(map)}.
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map) {
        return getDefaultFactory().newImmutableMap(map);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Map,
     * Map) newImmutableMap(map1, map2)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2) {
        return getDefaultFactory().newImmutableMap(map1, map2);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Map, Map,
     * Map) newImmutableMap(map1, map2, map3)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Map, Map,
     * Map, Map) newImmutableMap(map1, map2, map3, map4)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4);
    }

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Map, Map,
     * Map, Map, Map) newImmutableMap(map1, map2, map3, map4, map5)}.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
    * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Map<Character, Long> map1,
            @Nonnull Map<Character, Long> map2,
            @Nonnull Map<Character, Long> map3,
            @Nonnull Map<Character, Long> map4,
            @Nonnull Map<Character, Long> map5) {
        return getDefaultFactory().newImmutableMap(map1, map2, map3, map4, map5);
    }

    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(
     * Consumer) newImmutableMap(entriesSupplier)}.
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
    * 
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(@Nonnull
            Consumer<com.koloboke.function.CharLongConsumer> entriesSupplier
            ) {
        return getDefaultFactory().newImmutableMap(entriesSupplier);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(char[], long[]
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull char[] keys, @Nonnull long[] values) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Character[],
     * Long[]) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Character[] keys, @Nonnull Long[] values) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMap(Iterable, Iterable
     * ) newImmutableMap(keys, values)}.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
    * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMap(
            @Nonnull Iterable<Character> keys, @Nonnull Iterable<Long> values
            ) {
        return getDefaultFactory().newImmutableMap(keys, values);
    }

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMapOf(char, long
     * ) newImmutableMapOf(k1, v1)}.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMapOf(
            char k1, long v1) {
        return getDefaultFactory().newImmutableMapOf(k1, v1);
    }

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMapOf(char, long,
     * char, long) newImmutableMapOf(k1, v1, k2, v2)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMapOf(
            char k1, long v1, char k2, long v2) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2);
    }

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMapOf(char, long,
     * char, long, char, long) newImmutableMapOf(k1, v1, k2, v2,
     * k3, v3)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3);
    }

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMapOf(char, long,
     * char, long, char, long, char, long
     * ) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3, char k4, long v4) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashCharLongMapFactory#newImmutableMapOf(char, long,
     * char, long, char, long, char, long,
     * char, long) newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5)}.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
    public static  HashCharLongMap newImmutableMapOf(
            char k1, long v1, char k2, long v2,
            char k3, long v3, char k4, long v4,
            char k5, long v5) {
        return getDefaultFactory().newImmutableMapOf(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }

    private HashCharLongMaps() {}
}
