/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.page;


/**
 * 
 * @See {@link RulerPainter} RulerPainter.RULER_MARGIN5
 * @author Toni Kalajainen
 */
public class MarginUtils {

    public static Margin marginOf(double controlRelative, double controlAbsolute, double diagramAbsolute) {
        return new Margin(controlRelative/100, controlAbsolute, diagramAbsolute);
    }

    /** No margin */
    public static final Margin NO_MARGIN = marginOf(0, 0, 0);
    /** 5% margin */
    public static final Margin MARGIN5 = marginOf(5, 0, 0);
    /** 2% margin */
    public static final Margin MARGIN2 = marginOf(2, 0, 0);
    /** 10mm margin */
    public static final Margin MARGIN_10mm = marginOf(0, 0, 10);

    
    /** No margins */
    public static final Margins NO_MARGINS = new Margins(NO_MARGIN, NO_MARGIN, NO_MARGIN, NO_MARGIN);
    /** 5% margin */
    public static final Margins MARGINS5 = new Margins(MARGIN5, MARGIN5, MARGIN5, MARGIN5);
    /** 2% margin */
    public static final Margins MARGINS2 = new Margins(MARGIN2, MARGIN2, MARGIN2, MARGIN2);
    /** 10mm margin */
    public static final Margins MARGINS_10mm = new Margins(MARGIN_10mm, MARGIN_10mm, MARGIN_10mm, MARGIN_10mm);

    /**
     * A margin.
     */
    public static class Margin {
        /** Relative margins 0..100 (in control coordinates) */
        public final double controlRelative;
        /** Margins in absolute values of control units */
        public final double controlAbsolute;
        /** Margins in absolute values of diaram units*/
        public final double diagramAbsolute;

        public Margin(double controlRelative, double controlAbsolute, double diagramAbsolute) {
            this.controlRelative = controlRelative;
            this.controlAbsolute = controlAbsolute;
            this.diagramAbsolute = diagramAbsolute;
        }

        @Override
        public String toString() {
            return "(" + controlRelative + " " + controlAbsolute + " " + diagramAbsolute + ")";
        }

        public boolean isZero() {
            return Math.abs(controlRelative) < Double.MIN_VALUE
            && Math.abs(controlAbsolute) < Double.MIN_VALUE
            && Math.abs(diagramAbsolute) < Double.MIN_VALUE;
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            long temp;
            temp = Double.doubleToLongBits(controlAbsolute);
            result = prime * result + (int) (temp ^ (temp >>> 32));
            temp = Double.doubleToLongBits(controlRelative);
            result = prime * result + (int) (temp ^ (temp >>> 32));
            temp = Double.doubleToLongBits(diagramAbsolute);
            result = prime * result + (int) (temp ^ (temp >>> 32));
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            Margin other = (Margin) obj;
            if (Double.doubleToLongBits(controlAbsolute) != Double.doubleToLongBits(other.controlAbsolute))
                return false;
            if (Double.doubleToLongBits(controlRelative) != Double.doubleToLongBits(other.controlRelative))
                return false;
            if (Double.doubleToLongBits(diagramAbsolute) != Double.doubleToLongBits(other.diagramAbsolute))
                return false;
            return true;
        }
    }

    /**
     * 4 margins: top, bottom, left, right
     */
    public static class Margins {

        public static final int TOP    = 1 << 0;
        public static final int BOTTOM = 1 << 1;
        public static final int LEFT   = 1 << 2;
        public static final int RIGHT  = 1 << 3;

        public final Margin top;
        public final Margin bottom;
        public final Margin left;
        public final Margin right;

        public Margins(Margin top, Margin bottom, Margin left, Margin right) {
            assert top != null;
            assert bottom != null;
            assert left != null;
            assert right != null;

            this.top = top;
            this.bottom = bottom;
            this.left = left;
            this.right = right;
        }

        /**
         * @param a bitmask consisting of the values {@link #TOP},
         *        {@link #BOTTOM}, {@link #LEFT} and {@link #RIGHT}
         * @param top
         * @return
         */
        public Margins withSide(int side, Margin m) {
            Margin t = ((side & TOP) != 0) ? m : top;
            Margin b = ((side & BOTTOM) != 0) ? m : bottom;
            Margin l = ((side & LEFT) != 0) ? m : left;
            Margin r = ((side & RIGHT) != 0) ? m : right;
            return new Margins(t, b, l, r);
        }

        public boolean isZero() {
            return top.isZero() && bottom.isZero() && left.isZero() && right.isZero();
        }

        @Override
        public String toString() {
            return "[top=" + top + ", bottom=" + bottom + ", left=" + left + ", right=" + right + "]";
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + bottom.hashCode();
            result = prime * result + left.hashCode();
            result = prime * result + right.hashCode();
            result = prime * result + top.hashCode();
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            Margins other = (Margins) obj;
            if (!top.equals(other.top))
                return false;
            if (!bottom.equals(other.bottom))
                return false;
            if (!left.equals(other.left))
                return false;
            if (!right.equals(other.right))
                return false;
            return true;
        }
    }

}
