package org.simantics.utils;

import java.io.File;
import java.io.IOException;

/**
 * @author Tuukka Lehtonen
 */
public interface FileService {

	public static interface DeleteOption {}

	public static class EffortOption implements DeleteOption {
		public final int maxTries;

		private EffortOption(int maxTries) {
			this.maxTries = maxTries;
		}

		public static EffortOption maxTries(int maxTries) {
			return new EffortOption(maxTries);
		}
	}

	@FunctionalInterface
	public static interface FileOperation {
		IOperation<Boolean, IOException> perform(File file);

		default void perform(File... files) {
			for (File f : files)
				perform(f);
		}
	}

	public static class DeleteOperation implements FileOperation {
		private final FileService service;
		private final DeleteOption[] options;

		public DeleteOperation(FileService service, DeleteOption... options) {
			this.service = service;
			this.options = options;
		}

		@Override
		public IOperation<Boolean, IOException> perform(File file) {
			return service.scheduleDeleteIfExists(file, options);
		}
	}

	/**
	 * Schedules a file to be deleted at some point in the future when the
	 * system allows it to be deleted if ever.
	 * 
	 * @param file
	 * @return an operation handle for tracking the progress of the deletion or
	 *         <code>null</code> if there's a deletion operation scheduled for
	 *         the provided file in the queue or the file does not exist.
	 */
	IOperation<Boolean, IOException> scheduleDeleteIfExists(File file, DeleteOption... options);

	/**
	 * @param options
	 *            the deletion options to be used by the returned FileOperation
	 * @return a FileOperation that runs
	 *         {@link #scheduleDeleteIfExists(File, DeleteOption...)} for any
	 *         files given to it
	 * @since 1.28.0
	 */
	default FileOperation deleteOperation(DeleteOption... options) {
		return new DeleteOperation(this, options);
	}

}
