/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.lang.reflect.Array;
import java.security.InvalidParameterException;
import java.util.Arrays;
import java.util.Calendar;

/**
 * Object type casting tools.
 * 
 * @author Toni Kalajainen
 * @author Hannu Niemist&ouml;
 */
public class ValueUtils {

    /**
     * Convert double value to a number class.
     * 
     * @param value
     * @param numberClass
     * @return
     * @throws ClassCastException
     */
    public static Number doubleToNumberClass(double value, Class<? extends Number> numberClass)
    throws ClassCastException
    {
        if (numberClass==Integer.class)
            return (int)value;
        if (numberClass==Byte.class)
            return (byte)value;
        if (numberClass==Float.class)
            return (float)value;
        if (numberClass==Short.class)
            return (short)value;
        if (numberClass==Long.class)
            return (long)value;
        if (numberClass==Double.class)
            return (double)value;
        throw new ClassCastException("Cannot convert to "+numberClass.getName());
    }

    /**
     * Possible object types:
     * 
     */

    static final Boolean True = new Boolean(true);
    static final Boolean False = new Boolean(false);

    public static void serialize(DataOutput stream, Object obj) throws IOException {
        try {
            Class<?> clazz = obj.getClass();
            if(clazz.isArray()) {
                if(obj instanceof double[]) {
                    double[] array = (double[])obj;
                    stream.writeByte(16);
                    stream.writeInt(array.length);
                    for(double v : array)
                        stream.writeDouble(v);
                }
                else if(obj instanceof float[]) {
                    float[] array = (float[])obj;
                    stream.writeByte(15);
                    stream.writeInt(array.length);
                    for(float v : array)
                        stream.writeFloat(v);
                }
                else if(obj instanceof int[]) {
                    int[] array = (int[])obj;
                    stream.writeByte(13);
                    stream.writeInt(array.length);
                    for(int v : array)
                        stream.writeInt(v);
                }
                else if(obj instanceof String[]) {
                    String[] array = (String[])obj;
                    stream.writeByte(17);
                    stream.writeInt(array.length);
                    for(String v : array)
                        stream.writeUTF(v);
                }
                else if(obj instanceof boolean[]) {
                    boolean[] array = (boolean[])obj;
                    stream.writeByte(11);
                    stream.writeInt(array.length);
                    for(boolean v : array)
                        stream.writeByte(v ? (byte)1 : (byte)0);
                }
                else if(obj instanceof byte[]) {
                    byte[] array = (byte[])obj;
                    stream.writeByte(12);
                    stream.writeInt(array.length);
                    for(byte v : array)
                        stream.writeByte(v);
                }
                else if(obj instanceof long[]) {
                    long[] array = (long[])obj;
                    stream.writeByte(14);
                    stream.writeInt(array.length);
                    for(long v : array)
                        stream.writeLong(v);
                }
                else {
                    System.out.println("ValueUtils.serialize failed (invalid type).");
                    throw new InvalidParameterException();
                }
            }
            else {
                if(clazz == Double.class) {
                    stream.writeByte(6);
                    stream.writeDouble((Double)obj);
                }
                else if(clazz == Float.class) {
                    stream.writeByte(5);
                    stream.writeDouble((Float)obj);
                }
                else if(clazz == Integer.class) {
                    stream.writeByte(3);
                    stream.writeInt((Integer)obj);
                }
                else if(clazz == String.class) {
                    stream.writeByte(7);
                    stream.writeUTF((String)obj);
                }
                else if(clazz == Boolean.class) {
                    stream.writeByte((Boolean)obj ? 1 : 0);
                }
                else if(clazz == Byte.class) {
                    stream.writeByte(2);
                    stream.writeByte((Byte)obj);
                }
                else if(clazz == Long.class) {
                    stream.writeByte(4);
                    stream.writeLong((Long)obj);
                }
                else {
                    System.out.println("ValueUtils.serialize failed (invalid type).");
                    throw new InvalidParameterException();
                }
            }
        } catch(IOException e) {
            System.out.println("ValueUtils.serialize failed (write failure).");
            e.printStackTrace();
            throw e;
        }
    }

    public static Object deserialize(DataInput stream) throws IOException {
        try {
            byte typeCode = stream.readByte();
            switch(typeCode) {
                case 0: return False;
                case 1: return True;
                case 2: return stream.readByte();
                case 3: return stream.readInt();
                case 4: return stream.readLong();
                case 5: return stream.readFloat();
                case 6: return stream.readDouble();
                case 7: return stream.readUTF();

                case 11: {
                    int length = stream.readInt();
                    boolean[] value = new boolean[length];
                    for(int i=0;i<length;++i)
                        value[i] = stream.readByte() != 0;
                    return value;
                }
                case 12: {
                    int length = stream.readInt();
                    byte[] value = new byte[length];
                    for(int i=0;i<length;++i)
                        value[i] = stream.readByte();
                    return value;
                }
                case 13: {
                    int length = stream.readInt();
                    int[] value = new int[length];
                    for(int i=0;i<length;++i)
                        value[i] = stream.readInt();
                    return value;
                }
                case 14: {
                    int length = stream.readInt();
                    long[] value = new long[length];
                    for(int i=0;i<length;++i)
                        value[i] = stream.readLong();
                    return value;
                }
                case 15: {
                    int length = stream.readInt();
                    float[] value = new float[length];
                    for(int i=0;i<length;++i)
                        value[i] = stream.readFloat();
                    return value;
                }
                case 16: {
                    int length = stream.readInt();
                    double[] value = new double[length];
                    for(int i=0;i<length;++i)
                        value[i] = stream.readDouble();
                    return value;
                }
                case 17: {
                    int length = stream.readInt();
                    String[] value = new String[length];
                    for(int i=0;i<length;++i)
                        value[i] = stream.readUTF();
                    return value;
                }

            }

            System.out.println("ValueUtils.deserialize failed (invalid type code).");
            throw new IOException("Invalid value data.");
        } catch(IOException e) {
            System.out.println("ValueUtils.deserialize failed (reading failure).");
            e.printStackTrace();
            throw e;
        }
    }






    /**
     * is object an array type
     * @param obj the object
     * @return true if it is object array (Integer[], Short[], ect)
     */
    public static boolean isArray(Object obj) {
        return obj.getClass().isArray();
    }

    /**
     * Get the array length
     * @param obj object
     * @return the length
     */
    public static int getArrayLength(Object obj) {
        return Array.getLength(obj);
    }

    /**
     * Type casts an array to double.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static double[] toDoubleArray(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            double scalar = toDoubleScalar(obj);
            return new double[] {scalar};
        }
        int len = getArrayLength(obj);
        double[] result = new double[len];
        // TODO add support for int[] double[] float[], ect..
        if (obj instanceof Calendar[])
        {
            Calendar[] array = (Calendar[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].getTimeInMillis();
            return result;
        }
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].doubleValue();
            return result;
        }
        if (obj instanceof int[])
        {
            int[] array = (int[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Double[])
        {
            Double[] array = (Double[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].doubleValue();
            return result;
        }
        if (obj instanceof double[])
        {
            return (double[])obj;
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].doubleValue();
            return result;
        }
        if (obj instanceof float[])
        {
            float[] array = (float[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Long[])
        {
            Long[] array = (Long[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].doubleValue();
            return result;
        }
        if (obj instanceof long[])
        {
            long[] array = (long[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].doubleValue();
            return result;
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].doubleValue();
            return result;
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            for (int i=0; i<len; i++)
                try {
                    result[i] = new Double(array[i]);
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
                return result;
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (array[i].booleanValue()?1:0);
            return result;
        }
        if (obj instanceof boolean[])
        {
            boolean[] array = (boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (array[i]?1:0);
            return result;
        }
        throw new TypeCastException(obj.getClass());
    }

    /**
     * Type casts an object to long.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static double toDoubleScalar(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            if (obj instanceof Number)
                return ((Number)obj).doubleValue();
            if (obj instanceof Calendar)
                return ((Calendar) obj).getTimeInMillis();
            if (obj instanceof String)
            {
                String str = (String) obj;
                try {
                    Double d = new Double(str);
                    return d.doubleValue();
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
            }
            if (obj instanceof Boolean) {
                return (((Boolean)obj).booleanValue())?1:0;
            }
            throw new TypeCastException(obj.getClass());
        }
        int len = getArrayLength(obj);
        if (len!=1)
            throw new TypeCastException("Expected length of array is 1");

        if (obj instanceof double[])
        {
            return ((double[])obj)[0];
        }
        if (obj instanceof int[])
        {
            int[] array = (int[]) obj;
            return array[0];
        }
        if (obj instanceof Calendar[])
        {
            Calendar[] array = (Calendar[]) obj;
            return array[0].getTimeInMillis();
        }
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            return array[0].doubleValue();
        }
        if (obj instanceof Double[])
        {
            Double[] array = (Double[]) obj;
            return array[0].doubleValue();
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            return array[0].doubleValue();
        }
        if (obj instanceof float[])
        {
            float[] array = (float[]) obj;
            return array[0];
        }
        if (obj instanceof Long[])
        {
            Long[] array = (Long[]) obj;
            return array[0].doubleValue();
        }
        if (obj instanceof long[])
        {
            long[] array = (long[]) obj;
            return array[0];
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            return array[0].doubleValue();
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            return array[0];
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            return array[0].doubleValue();
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            return array[0];
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            try {
                return new Double(array[0]);
            } catch (NumberFormatException e) {
                throw new TypeCastException(obj.getClass());
            }
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            return (array[0].booleanValue()?1:0);
        }
        if (obj instanceof boolean[])
        {
            boolean[] array = (boolean[]) obj;
            return (array[0]?1:0);
        }
        throw new TypeCastException(obj.getClass());
    }


    /**
     * Type casts an array to float.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static float[] toFloatArray(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            float scalar = toFloatScalar(obj);
            return new float[] {scalar};
        }
        int len = getArrayLength(obj);
        float[] result = new float[len];
        // TODO add support for int[] float[] float[], ect..
        if (obj instanceof Calendar[])
        {
            Calendar[] array = (Calendar[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].getTimeInMillis();
            return result;
        }
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].floatValue();
            return result;
        }
        if (obj instanceof int[])
        {
            int[] array = (int[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].floatValue();
            return result;
        }
        if (obj instanceof float[])
        {
            return (float[])obj;
        }
        if (obj instanceof Double[])
        {
            Double[] array = (Double[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].floatValue();
            return result;
        }
        if (obj instanceof double[])
        {
            double[] array = (double[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (float)array[i];
            return result;
        }
        if (obj instanceof Long[])
        {
            Long[] array = (Long[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].floatValue();
            return result;
        }
        if (obj instanceof long[])
        {
            long[] array = (long[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].floatValue();
            return result;
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].floatValue();
            return result;
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            for (int i=0; i<len; i++)
                try {
                    result[i] = new Float(array[i]);
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
                return result;
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (array[i].booleanValue()?1:0);
            return result;
        }
        if (obj instanceof boolean[])
        {
            boolean[] array = (boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (array[i]?1:0);
            return result;
        }
        throw new TypeCastException(obj.getClass());
    }

    /**
     * Type casts an object to float.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static float toFloatScalar(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            if (obj instanceof Number)
                return ((Number)obj).floatValue();
            if (obj instanceof Calendar)
                return ((Calendar) obj).getTimeInMillis();
            if (obj instanceof String)
            {
                String str = (String) obj;
                try {
                    Float d = new Float(str);
                    return d.floatValue();
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
            }
            if (obj instanceof Boolean) {
                return (((Boolean)obj).booleanValue())?1:0;
            }
            throw new TypeCastException(obj.getClass());
        }
        int len = getArrayLength(obj);
        if (len!=1)
            throw new TypeCastException("Expected length of array is 1");

        if (obj instanceof float[])
        {
            return ((float[])obj)[0];
        }
        if (obj instanceof int[])
        {
            int[] array = (int[]) obj;
            return array[0];
        }
        if (obj instanceof Calendar[])
        {
            Calendar[] array = (Calendar[]) obj;
            return array[0].getTimeInMillis();
        }
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            return array[0].floatValue();
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            return array[0].floatValue();
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            return array[0].floatValue();
        }
        if (obj instanceof double[])
        {
            double[] array = (double[]) obj;
            return (float)array[0];
        }
        if (obj instanceof Long[])
        {
            Long[] array = (Long[]) obj;
            return array[0].floatValue();
        }
        if (obj instanceof long[])
        {
            long[] array = (long[]) obj;
            return array[0];
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            return array[0].floatValue();
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            return array[0];
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            return array[0].floatValue();
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            return array[0];
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            try {
                return new Float(array[0]);
            } catch (NumberFormatException e) {
                throw new TypeCastException(obj.getClass());
            }
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            return (array[0].booleanValue()?1:0);
        }
        if (obj instanceof boolean[])
        {
            boolean[] array = (boolean[]) obj;
            return (array[0]?1:0);
        }
        throw new TypeCastException(obj.getClass());
    }


    /**
     * Type casts an object to long.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static long toLongScalar(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            if (obj instanceof Number)
                return ((Number)obj).longValue();
            if (obj instanceof Calendar)
                return ((Calendar) obj).getTimeInMillis();
            if (obj instanceof String)
            {
                String str = (String) obj;
                try {
                    Long d = new Long(str);
                    return d.longValue();
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
            }
            if (obj instanceof Boolean) {
                return (((Boolean)obj).booleanValue())?1:0;
            }
            throw new TypeCastException(obj.getClass());
        }
        int len = getArrayLength(obj);
        if (len!=1)
            throw new TypeCastException("Expected length of array is 1");

        if (obj instanceof double[])
        {
            return ((long[])obj)[0];
        }
        if (obj instanceof int[])
        {
            int[] array = (int[]) obj;
            return array[0];
        }
        if (obj instanceof Calendar[])
        {
            Calendar[] array = (Calendar[]) obj;
            return array[0].getTimeInMillis();
        }
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            return array[0].longValue();
        }
        if (obj instanceof Double[])
        {
            Double[] array = (Double[]) obj;
            return array[0].longValue();
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            return array[0].longValue();
        }
        if (obj instanceof float[])
        {
            float[] array = (float[]) obj;
            return (long)array[0];
        }
        if (obj instanceof Long[])
        {
            Long[] array = (Long[]) obj;
            return array[0].longValue();
        }
        if (obj instanceof long[])
        {
            long[] array = (long[]) obj;
            return array[0];
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            return array[0].longValue();
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            return array[0];
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            return array[0].longValue();
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            return array[0];
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            try {
                return new Long(array[0]);
            } catch (NumberFormatException e) {
                throw new TypeCastException(obj.getClass());
            }
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            return (array[0].booleanValue()?1:0);
        }
        if (obj instanceof boolean[])
        {
            boolean[] array = (boolean[]) obj;
            return (array[0]?1:0);
        }
        throw new TypeCastException(obj.getClass());
    }

    /**
     * Type casts an array to long.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static long[] toLongArray(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            long scalar = toLongScalar(obj);
            return new long[] {scalar};
        }
        int len = getArrayLength(obj);
        long[] result = new long[len];
        // TODO add support for int[] long[] float[], ect..
        if (obj instanceof Calendar[])
        {
            Calendar[] array = (Calendar[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].getTimeInMillis();
            return result;
        }
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].longValue();
            return result;
        }
        if (obj instanceof int[])
        {
            int[] array = (int[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Long[])
        {
            Long[] array = (Long[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].longValue();
            return result;
        }
        if (obj instanceof long[])
        {
            return (long[])obj;
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].longValue();
            return result;
        }
        if (obj instanceof float[])
        {
            float[] array = (float[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (long)array[i];
            return result;
        }
        if (obj instanceof Double[])
        {
            Double[] array = (Double[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].longValue();
            return result;
        }
        if (obj instanceof double[])
        {
            double[] array = (double[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (long)array[i];
            return result;
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].longValue();
            return result;
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].longValue();
            return result;
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            for (int i=0; i<len; i++)
                try {
                    result[i] = new Long(array[i]);
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
                return result;
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (array[i].booleanValue()?1:0);
            return result;
        }
        if (obj instanceof boolean[])
        {
            boolean[] array = (boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (array[i]?1:0);
            return result;
        }
        throw new TypeCastException(obj.getClass());
    }






    /**
     * Type casts an object to int.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static int toIntegerScalar(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            if (obj instanceof Number)
                return ((Number)obj).intValue();
            if (obj instanceof String)
            {
                String str = (String) obj;
                try {
                    Integer d = new Integer(str);
                    return d.intValue();
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
            }
            if (obj instanceof Boolean) {
                return (((Boolean)obj).booleanValue())?1:0;
            }
            throw new TypeCastException(obj.getClass());
        }
        int len = getArrayLength(obj);
        if (len!=1)
            throw new TypeCastException("Expected length of array is 1");

        if (obj instanceof double[])
        {
            return ((int[])obj)[0];
        }
        if (obj instanceof int[])
        {
            int[] array = (int[]) obj;
            return array[0];
        }
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            return array[0].intValue();
        }
        if (obj instanceof Double[])
        {
            Double[] array = (Double[]) obj;
            return array[0].intValue();
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            return array[0].intValue();
        }
        if (obj instanceof float[])
        {
            float[] array = (float[]) obj;
            return (int)array[0];
        }
        if (obj instanceof Long[])
        {
            Long[] array = (Long[]) obj;
            return array[0].intValue();
        }
        if (obj instanceof long[])
        {
            long[] array = (long[]) obj;
            return (int)array[0];
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            return array[0].intValue();
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            return array[0];
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            return array[0].intValue();
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            return array[0];
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            try {
                return new Integer(array[0]);
            } catch (NumberFormatException e) {
                throw new TypeCastException(obj.getClass());
            }
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            return (array[0].booleanValue()?1:0);
        }
        if (obj instanceof boolean[])
        {
            boolean[] array = (boolean[]) obj;
            return (array[0]?1:0);
        }
        throw new TypeCastException(obj.getClass());
    }

    /**
     * Type casts an array to int.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static int[] toIntegerArray(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            int scalar = toIntegerScalar(obj);
            return new int[] {scalar};
        }
        int len = getArrayLength(obj);
        int[] result = new int[len];
        // TODO add support for int[] int[] float[], ect..
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].intValue();
            return result;
        }
        if (obj instanceof long[])
        {
            long[] array = (long[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (int)array[i];
            return result;
        }
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].intValue();
            return result;
        }
        if (obj instanceof int[])
        {
            return (int[])obj;
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].intValue();
            return result;
        }
        if (obj instanceof float[])
        {
            float[] array = (float[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (int)array[i];
            return result;
        }
        if (obj instanceof Double[])
        {
            Double[] array = (Double[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].intValue();
            return result;
        }
        if (obj instanceof double[])
        {
            double[] array = (double[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (int)array[i];
            return result;
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].intValue();
            return result;
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].intValue();
            return result;
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i];
            return result;
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            for (int i=0; i<len; i++)
                try {
                    result[i] = new Integer(array[i]);
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
                return result;
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (array[i].booleanValue()?1:0);
            return result;
        }
        if (obj instanceof boolean[])
        {
            boolean[] array = (boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = (array[i]?1:0);
            return result;
        }
        throw new TypeCastException(obj.getClass());
    }




    /**
     * Type casts an array to boolean.
     *
     * 0 is converted to false, 1 is converted to true, others throw TypeCastException
     * 
     * @return type casted value
     * @throws TypeCastException
     */
    public static boolean[] toBooleanArray(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            boolean scalar = toBooleanScalar(obj);
            return new boolean[] {scalar};
        }
        int len = getArrayLength(obj);
        boolean[] result = new boolean[len];
        if (obj instanceof Integer[])
        {
            Integer[] array = (Integer[]) obj;
            for (int i=0; i<len; i++) {
                int value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof int[])
        {
            int[] array = (int[]) obj;
            for (int i=0; i<len; i++) {
                int value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            for (int i=0; i<len; i++)
                result[i] = array[i].booleanValue();
            return result;
        }
        if (obj instanceof boolean[])
        {
            return (boolean[])obj;
        }
        if (obj instanceof Float[])
        {
            Float[] array = (Float[]) obj;
            for (int i=0; i<len; i++) {
                float value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof float[])
        {
            float[] array = (float[]) obj;
            for (int i=0; i<len; i++) {
                float value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof Long[])
        {
            Long[] array = (Long[]) obj;
            for (int i=0; i<len; i++) {
                long value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof long[])
        {
            long[] array = (long[]) obj;
            for (int i=0; i<len; i++) {
                long value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof Short[])
        {
            Short[] array = (Short[]) obj;
            for (int i=0; i<len; i++) {
                short value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof short[])
        {
            short[] array = (short[]) obj;
            for (int i=0; i<len; i++) {
                short value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof Byte[])
        {
            Byte[] array = (Byte[]) obj;
            for (int i=0; i<len; i++) {
                byte value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof byte[])
        {
            byte[] array = (byte[]) obj;
            for (int i=0; i<len; i++) {
                byte value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            for (int i=0; i<len; i++)
                try {
                    result[i] = new Boolean(array[i]);
                } catch (NumberFormatException e) {
                    throw new TypeCastException(obj.getClass());
                }
                return result;
        }
        if (obj instanceof Double[])
        {
            Double[] array = (Double[]) obj;
            for (int i=0; i<len; i++) {
                double value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        if (obj instanceof double[])
        {
            double[] array = (double[]) obj;
            for (int i=0; i<len; i++) {
                double value = array[i];
                if (value==0) result[i] = false;
                else if (value==1) result[i] = true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            return result;
        }
        throw new TypeCastException(obj.getClass());
    }

    /**
     * Type casts an object to long.
     * 
     * String types are attempted to be converted. If string
     * doesn't represent numeric value, 0 will be returned
     * 
     * boolean types return 0/1, for false/true respectively
     * 
     * @return type casted value
     */
    public static boolean toBooleanScalar(Object obj)
    throws TypeCastException
    {
        if (!isArray(obj)) {
            if (obj instanceof Boolean) {
                return ((Boolean)obj).booleanValue();
            }
            if (obj instanceof Byte) {
                byte value = (Byte)obj;
                if (value==0) return false;
                else if (value==1) return true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            if (obj instanceof Short) {
                short value = (Short)obj;
                if (value==0) return false;
                else if (value==1) return true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            if (obj instanceof Integer) {
                int value = (Integer)obj;
                if (value==0) return false;
                else if (value==1) return true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            if (obj instanceof Long) {
                long value = (Long)obj;
                if (value==0) return false;
                else if (value==1) return true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            if (obj instanceof Float) {
                float value = (Float)obj;
                if (value==0) return false;
                else if (value==1) return true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            if (obj instanceof Double) {
                double value = (Double)obj;
                if (value==0) return false;
                else if (value==1) return true;
                else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            if (obj instanceof String) {
                String value = (String)obj;
                if (value.equals("true") || value.equals("1")) return true;
                if (value.equals("false") || value.equals("0")) return false;
                throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
            }
            throw new TypeCastException(obj.getClass());
        }
        int len = getArrayLength(obj);
        if (len!=1)
            throw new TypeCastException("Expected length of array is 1");

        if (obj instanceof boolean[])
        {
            return ((boolean[])obj)[0];
        }
        if (obj instanceof Boolean[])
        {
            Boolean[] array = (Boolean[]) obj;
            return array[0].booleanValue();
        }
        if (obj instanceof int[])
        {
            int value = ((int[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof Integer[])
        {
            int value = ((Integer[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof Float[])
        {
            float value = ((Float[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof float[])
        {
            float value = ((float[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof Long[])
        {
            long value = ((Long[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof long[])
        {
            long value = ((long[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof Short[])
        {
            Short value = ((Short[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof short[])
        {
            short value = ((short[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof Byte[])
        {
            Byte value = ((Byte[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof byte[])
        {
            byte value = ((byte[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof String[])
        {
            String[] array = (String[]) obj;
            String value = array[0];
            if (value.equals("true") || value.equals("1")) return true;
            if (value.equals("false") || value.equals("0")) return false;
            throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof Double[])
        {
            double value = ((Double[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        if (obj instanceof double[])
        {
            double value = ((double[])obj)[0];
            if (value==0) return false;
            else if (value==1) return true;
            else throw new TypeCastException("Cannot convert "+value+" to boolean value - 0 or 1 is expected");
        }
        throw new TypeCastException(obj.getClass());
    }


    @SuppressWarnings("unchecked")
    public static <T> T adaptToClass(Object obj, Class<T> clazz)
    throws TypeCastException
    {
        if (clazz.equals(double[].class))
            return (T) toDoubleArray(obj);
        if (clazz.equals(Double.class))
            return (T) new Double(toDoubleScalar(obj));

        if (clazz.equals(float[].class))
            return (T) toFloatArray(obj);
        if (clazz.equals(Float.class))
            return (T) new Float(toFloatScalar(obj));

        if (clazz.equals(long[].class))
            return (T) toLongArray(obj);
        if (clazz.equals(Long.class))
            return (T) new Long(toLongScalar(obj));

        if (clazz.equals(int[].class))
            return (T) toIntegerArray(obj);
        if (clazz.equals(Integer.class))
            return (T) new Integer(toIntegerScalar(obj));

        if (clazz.equals(boolean[].class))
            return (T) toBooleanArray(obj);
        if (clazz.equals(Boolean.class))
            return (T) new Long(toLongScalar(obj));

        throw new TypeCastException("Unsupported conversion type from "+obj.getClass()+" to "+clazz);
    }



    public static String toString(Object obj) {
        if (obj instanceof Object[]) {
            return Arrays.toString((Object[])obj);
        }
        if (obj instanceof double[]) return Arrays.toString((double[])obj);
        if (obj instanceof boolean[]) return Arrays.toString((boolean[])obj);
        if (obj instanceof byte[]) return Arrays.toString((byte[])obj);
        if (obj instanceof char[]) return Arrays.toString((char[])obj);
        if (obj instanceof float[]) return Arrays.toString((float[])obj);
        if (obj instanceof int[]) return Arrays.toString((int[])obj);
        if (obj instanceof long[]) return Arrays.toString((long[])obj);
        if (obj instanceof short[]) return Arrays.toString((short[])obj);
        return obj.toString();
    }
}
