/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * MapSet is an associative data structure a key type on the left side (L) and a
 * set element type of the right side.
 * 
 * <p>
 * Based on {@link MapList} by Toni Kalajainen.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public abstract class MapSet<L, R> {

    protected Map<L, Set<R>> sets;
    
    public static class Hash<L, R> extends MapSet<L, R> {
    	public Hash() {
    		sets = new HashMap<L, Set<R>>();
    	}
        protected Set<R> getOrCreateSet(L key) {
            Set<R> set = sets.get(key);
            if (set == null) {
                set = new HashSet<R>();
                sets.put(key, set);
            }
            return set;
        }
    }

    public static class Tree<L, R> extends MapSet<L, R> {
    	public Tree() {
    		sets = new TreeMap<L, Set<R>>();
    	}
    	public Tree(Comparator<? super L> comparator) {
    		sets = new TreeMap<L, Set<R>>(comparator);
    	}
        protected Set<R> getOrCreateSet(L key) {
            Set<R> set = sets.get(key);
            if (set == null) {
                set = new HashSet<R>();
                sets.put(key, set);
            }
            return set;
        }
    }
    
    public boolean add(L key, R value) {
        Set<R> set = getOrCreateSet(key);
        return set.add(value);
    }

    protected abstract Set<R> getOrCreateSet(L key);

    private Set<R> getSet(L key) {
        return sets.get(key);
    }

    /**
     * @param key
     * @return a valid set, empty if no values exist
     */
    public Set<R> removeValues(L key) {
        Set<R> set = sets.remove(key);
        if (set == null)
            return Collections.emptySet();
        return set;
    }

    public boolean remove(L key, R value) {
        Set<R> set = getSet(key);
        if (set == null)
            return false;
        boolean result = set.remove(value);
        if (set.isEmpty())
            sets.remove(key);
        return result;
    }

    public void clear() {
        sets.clear();
    }

    public L[] getKeys(L[] list) {
        return sets.keySet().toArray(list);
    }

    public Set<L> getKeys() {
        return sets.keySet();
    }

    public boolean hasValues(L key) {
        return sets.containsKey(key);
    }

    public R[] getValues(L key, R[] list) {
        Set<R> l = sets.get(key);
        if (l == null)
            return null;
        return l.toArray(list);
    }

    public Set<R> getValues(L key) {
        Set<R> l = sets.get(key);
        if (l == null)
            return null;
        return new HashSet<R>(l);
    }

    public Set<R> getValuesUnsafe(L key) {
        return sets.get(key);
    }

}
