/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench.handler.e4;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.eclipse.core.commands.Command;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.State;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.e4.core.contexts.Active;
import org.eclipse.e4.core.di.annotations.CanExecute;
import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.ui.model.application.MApplication;
import org.eclipse.e4.ui.model.application.ui.basic.MPart;
import org.eclipse.e4.ui.workbench.modeling.EModelService;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.commands.ICommandService;
import org.simantics.Simantics;
import org.simantics.db.ChangeSetIdentifier;
import org.simantics.db.Operation;
import org.simantics.db.Session;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.NoHistoryException;
import org.simantics.db.service.ManagementSupport;
import org.simantics.db.service.UndoRedoSupport;
import org.simantics.ui.states.TrackedTextState;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * @author Kalle Kondelin
 * @author Jani Simomaa
 */
public class SessionUndoHandler {
    
    private final static boolean DEBUG = false;
    private final static boolean ENABLED = true;
    
    @CanExecute
    public boolean canExecute() {
        return UndoRedoTester.canUndo();
    }
    
    @Execute
    public void execute(@Active MPart activePart, MApplication mApplication, EModelService modelService) throws ExecutionException {
        if (DEBUG)
            System.out.println("--\nUndo handler called.");

        if (activePart == null)
            return;
        final Session session = Simantics.peekSession();
        if (session == null)
            return;
        
        ICommandService service = (ICommandService) PlatformUI.getWorkbench().getService(ICommandService.class);
        Command command = service.getCommand( TrackedTextState.COMMAND_ID );
        State state = command.getState( TrackedTextState.STATE_ID );
        
        boolean sessionUndoEnabled = true;
        Object value = state.getValue();
        if (value != null && value instanceof Boolean)
            sessionUndoEnabled = (Boolean) value;
        
        if (ENABLED) {
            if (sessionUndoEnabled) {
                try {
                    final AtomicReference<String> msg = new AtomicReference<String>();
                    IRunnableWithProgress undo = new IRunnableWithProgress() {
                        @Override
                        public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                            try {
                                monitor.beginTask("Undo", IProgressMonitor.UNKNOWN);
                                msg.set( undo( session ) );
                            } catch (NoHistoryException e) {
                                msg.set("Nothing to undo.");
                            } catch (DatabaseException e) {
                                throw new InvocationTargetException(e);
                            } finally {
                                monitor.done();
                            }
                        }
                    };
    
                    // busyCursorWhile does not work because locking the session for undo
                    // will also lock the UI completely.
                    //PlatformUI.getWorkbench().getProgressService().busyCursorWhile(undo);
                    Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
                    new ProgressMonitorDialog(shell).run(true, false, undo);
                    
                    //
                    // TODO Not the Eclipse 4 way of using IStatusLineManager!
                    // See:
                    // https://bugs.eclipse.org/bugs/show_bug.cgi?id=332499
                    // https://www.eclipse.org/forums/index.php?t=msg&th=367379&goto=941232&#msg_941232
                    //
                    IWorkbenchPart part = WorkbenchUtils.getActiveWorkbenchPart();
                    IStatusLineManager manager = WorkbenchUtils.getStatusLine(part);
                    manager.setMessage(msg.get());
//                    statusManager.setMessage( msg.get() );
                } catch (InvocationTargetException e1) {
                    throw new ExecutionException("Undo failed, database failure.", e1.getCause());
                } catch (InterruptedException e1) {
                    throw new ExecutionException("Undo failed, interrupted.", e1);
                }
            }
        } else {
            ErrorLogger.getDefault().log(IStatus.INFO, 0, "The undo command is disabled", null);
        }
        return;
    }

    public static String getComment(Session session, ChangeSetIdentifier id) {
        byte[] data = id.getMetadata().get(CommentMetadata.class.getName());
        if(data == null)
            return "Undescribed operation.";
        String comment = CommentMetadata.deserialise(session, data).toString().trim();
        if(comment.isEmpty())
            return "Undescribed operation.";
        return comment;
    }
    
    private String undo(Session session) throws DatabaseException {
        UndoRedoSupport support = session.getService(UndoRedoSupport.class);
        
        List<Operation> ops = support.undoAndReturnOperations(session, 1);
        if(ops.isEmpty())
            return "Undo history is empty.";        
        
        Operation mainOperation = ops.get(0);
        
        String msg = null;
        long csId = mainOperation.getCSId();

        ManagementSupport management = session.getService(ManagementSupport.class);
        for(ChangeSetIdentifier id : management.getChangeSetIdentifiers(csId, csId))
            if(msg == null)
                msg = "Undo reverted: " + getComment(session, id);

        if (DEBUG)
            System.out.println(msg);
        return msg;
    }

}
