/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;
import org.simantics.ui.internal.Activator;
import org.simantics.utils.datastructures.MapList;

public class PerspectiveContextBindingManager implements IExtensionChangeHandler {

    private final static String                           NAMESPACE  = Activator.PLUGIN_ID;

    private final static String                           EP_NAME    = "perspectiveContextBinding";

    private ExtensionTracker                              tracker;

    private MapList<String, IPerspectiveContextExtension> extensions = new MapList<String, IPerspectiveContextExtension>();

    private static PerspectiveContextBindingManager       INSTANCE;

    public static synchronized PerspectiveContextBindingManager getInstance() {
        if (INSTANCE == null)
            INSTANCE = new PerspectiveContextBindingManager();
        return INSTANCE;
    }

    public static synchronized void dispose() {
        if (INSTANCE != null) {
            INSTANCE.close();
            INSTANCE = null;
        }
    }

    private PerspectiveContextBindingManager() {
        tracker = new ExtensionTracker();

        // Cache defined actions
        IExtensionPoint expt = Platform.getExtensionRegistry().getExtensionPoint(NAMESPACE, EP_NAME);
        loadExtensions(expt.getConfigurationElements());

        // Start tracking for new and removed extensions
        IFilter filter = ExtensionTracker.createExtensionPointFilter(expt);
        tracker.registerHandler(this, filter);
    }

    private void close() {
        tracker.close();
        tracker = null;
        extensions = new MapList<String, IPerspectiveContextExtension>();
    }

    public synchronized List<IPerspectiveContextExtension> getExtensions(String perspectiveId) {
        List<IPerspectiveContextExtension> exts = extensions.getValues(perspectiveId);
        if (exts == null)
            return Arrays.asList();
        return Collections.unmodifiableList(exts);
    }

    private synchronized void loadExtensions(IConfigurationElement[] elements) {
        for (IConfigurationElement el : elements) {
            String perspectiveId = el.getAttribute("perspectiveId");
            String contextId = el.getAttribute("contextIds");
            String[] contextIds = contextId.split(",");

            IPerspectiveContextExtension ext = new IPerspectiveContextExtension.Stub(perspectiveId, contextIds);

            // Start tracking the new extension object, its removal will be notified of
            // with removeExtension(extension, Object[]).
            tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

            extensions.add(perspectiveId, ext);
        }
    }

    @Override
    public void addExtension(IExtensionTracker tracker, IExtension extension) {
        loadExtensions(extension.getConfigurationElements());
    }

    @Override
    public synchronized void removeExtension(IExtension extension, Object[] objects) {
       
        for (Object o : objects) {
            IPerspectiveContextExtension ext = (IPerspectiveContextExtension) o;
            tracker.unregisterObject(extension, ext);
            extensions.remove(ext.getPerspectiveId(), ext);
        }

    }

}
