/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.tester;

import java.util.Arrays;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.expressions.PropertyTester;
import org.simantics.DatabaseJob;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleTypedParent;
import org.simantics.db.common.request.Queries;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.common.utils.RequestUtil;
import org.simantics.db.exception.BindingException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.exception.ValueTypeMismatchException;
import org.simantics.db.management.ISessionContext;
import org.simantics.layer0.Layer0;
import org.simantics.scl.reflection.OntologyVersions;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.utils.ResourceAdaptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class BasicResourcePropertyTester extends PropertyTester {

    private static final boolean DEBUG = false;

    /**
     * Tests if the received resource is an instance of any of the URIs
     * listed in the arguments.
     */
    protected static final String RESOURCE_TYPE = "resourceType";

    /**
     * Tests if the received resource(s) are all instances of any of the URIs
     * listed in the arguments.
     */
    protected static final String ALL_RESOURCES_OF_TYPE = "allResourcesOfType";

    /**
     * Tests if the received resource inherits any of the URIs
     * listed in the arguments.
     */
    protected static final String INHERITS = "inherits";

    /**
     * Tests if all the received resource(s) inherit any of the URIs
     * listed in the arguments.
     */
    protected static final String ALL_INHERIT = "allInherit";

    /**
     * Tests if the received resource has a name property that matches any of
     * the regular expressions specified in the arguments.
     */
    protected static final String NAME_MATCHES = "nameMatches";

    /**
     * Tests if the received resource has any of the tags listed as URIs in the
     * arguments.
     */
    protected static final String HAS_TAG = "hasTag";

    /**
     * Tests if the received resource has any of the tags listed as URIs in the
     * arguments.
     */
    protected static final String HAS_PARENT = "hasParent";
    
    /**
     * Tests if the received resource has any of the statements with the predicate type listed as URI in the
     * arguments.
     */
    protected static final String HAS_STATEMENT = "hasStatement";
    
    private static final OntologyVersions VERSIONS = OntologyVersions.getInstance();

    @Override
    public boolean test(final Object receiver, final String property, final Object[] args, final Object expectedValue) {
        if (DEBUG)
            System.out.println("TEST: " + receiver + ", " + property + ", " + Arrays.toString(args) + ", " + expectedValue);

        // Receiver sanity check related to tested property
        final Resource resource = ResourceAdaptionUtils.toSingleResource(receiver);
        final Resource[] resources = ResourceAdaptionUtils.toResources(receiver);
        if (resources.length == 0)
            return false;

        // Don't test with multi-selection and single resource test.
        final boolean multiResourceTest = ALL_RESOURCES_OF_TYPE.equals(property) || ALL_INHERIT.equals(property);
        if (!multiResourceTest && resource == null)
            return false;

        if (DatabaseJob.inProgress())
            return false;

        ISessionContext ctx = Simantics.getSessionContext();
        if (ctx == null)
            return false;

        try {
            return RequestUtil.trySyncRequest(
                    ctx.getSession(),
                    SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                    SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                    false,
                    new UniqueRead<Boolean>() {
                @Override
                public Boolean perform(ReadGraph g) throws DatabaseException {
                    if (multiResourceTest) {
                        return Boolean.valueOf(doTest(g, resources, property, args, expectedValue));
                    } else {
                        return Boolean.valueOf(doTest(g, resource, property, args, expectedValue));
                    }
                }
            });
        } catch (DatabaseException | InterruptedException e) {
            // Purposefully not logging these exceptions, there might be way too
            // many even under normal circumstances.
            // TODO: add debug tracing options controlling the printing of these exceptions
            return false;
        }
    }

    protected boolean doTest(ReadGraph g, Resource resource, String property, Object[] args, Object expectedValue) throws DatabaseException {
        if (RESOURCE_TYPE.equals(property)) {
            if (DEBUG)
                System.out.println("** " + NameUtils.getSafeName(g, resource));
            try {
                for (int i = 0; i < args.length; i++) {
                    if (g.isInstanceOf(resource, g.getResource(VERSIONS.currentVersion((String) args[i])))) {
                        if (DEBUG)
                            System.out.println("IS INSTANCE OF " + args[i]);
                        return true;
                    }
                }
            } catch (ResourceNotFoundException e) {
                /* This is a natural situation (database does not contain all resources
                   referred in plugins). No error reporting.
                 */
                return false;
            }
            if (DEBUG)
                System.out.println("NOT AN INSTANCE OF ANY OF: " + Arrays.toString(args));
            return false;
        } else if (INHERITS.equals(property)) {
            if (DEBUG)
                System.out.println("** " + NameUtils.getSafeName(g, resource));
            try {
                for (int i = 0; i < args.length; i++) {
                    if (g.isInheritedFrom(resource, g.getResource(VERSIONS.currentVersion((String) args[i])))) {
                        if (DEBUG)
                            System.out.println("INHERITS " + args[i]);
                        return true;
                    }
                }
            } catch (ResourceNotFoundException e) {
                /* This is a natural situation (database does not contain all resources
                   referred in plugins). No error reporting.
                 */
                return false;
            }
            if (DEBUG)
                System.out.println("DOES NOT INHERIT ANY OF: " + Arrays.toString(args));
            return false;
        } else if (NAME_MATCHES.equals(property)) {
            if (args.length == 0)
                return false;
            Pattern[] patterns = new Pattern[args.length];
            for (int i = 0; i < args.length; i++) {
                patterns[i] = Pattern.compile((String) args[i]);
            }
            try {
                Layer0 L0 = Layer0.getInstance(g);
                for (Resource r : g.getObjects(resource, L0.HasName)) {
                    String name = g.getPossibleValue(r, Bindings.STRING);
                    if (name == null)
                        continue;
                    for (Pattern p : patterns) {
                        Matcher m = p.matcher(name);
                        if (m.matches())
                            return true;
                    }
                }
            } catch (ValueTypeMismatchException e) {
                e.printStackTrace();
            } catch (BindingException e) {
                e.printStackTrace();
            }
            return false;
        } else if (HAS_TAG.equals(property)) {
            try {
                for (int i = 0; i < args.length; i++) {
                    Resource tag = g.syncRequest(Queries.resource((String) args[i]));
                    if (g.hasStatement(resource, tag)) {
                        if (DEBUG)
                            System.out.println("HAS TAG " + args[i]);
                        return true;
                    }
                }
            } catch (ResourceNotFoundException e) {
                /* This is a natural situation (database does not contain all resources
                   referred in plugins). No error reporting.
                 */
                return false;
            }
        } else if (HAS_PARENT.equals(property)) {
            try {
                for (int i = 0; i < args.length; i++) {
                    Resource type = g.syncRequest(Queries.resource((String) args[i]));
                    Resource parent = g.syncRequest(new PossibleTypedParent(resource, type));
                    if(parent != null) {
                        if (DEBUG)
                            System.out.println("HAS PARENT " + args[i]);
                        return true;
                    }
                }
            } catch (ResourceNotFoundException e) {
                /* This is a natural situation (database does not contain all resources
                   referred in plugins). No error reporting.
                 */
                return false;
            }
        } else if (HAS_STATEMENT.equals(property)) {
            try {
                for (int i = 0; i < args.length; i++) {
                    Resource predicate = g.getResource(VERSIONS.currentVersion((String) args[i]));
                    if (g.hasStatement(resource, predicate)) {
                        if (DEBUG)
                            System.out.println("HAS STATEMENT " + args[i]);
                        return true;
                    }
                }
            } catch (ResourceNotFoundException e) {
                /* This is a natural situation (database does not contain all resources
                   referred in plugins). No error reporting.
                 */
                return false;
            }
        }
        return false;
    }

    protected boolean doTest(ReadGraph g, Resource[] resources, String property, Object[] args, Object expectedValue) throws DatabaseException {
        if (ALL_RESOURCES_OF_TYPE.equals(property)) {
            Resource[] argTypes = new Resource[args.length];
            boolean argTypesResolved = false;
            for (int i = 0; i < args.length; i++) {
                try {
                    argTypes[i] = g.getResource(VERSIONS.currentVersion((String) args[i]));
                    argTypesResolved = true;
                } catch (ResourceNotFoundException e) {
                    /* This is a natural situation (database does not contain all resources
                           referred in plugins). No error reporting.
                     */
                }
            }
            if (!argTypesResolved) {
                if (DEBUG)
                    System.out.println("(WW) NO ARGUMENTS RESOLVED INTO RESOURCES!");
                return false;
            }

            for (Resource resource : resources) {
                if (DEBUG)
                    System.out.println("** " + NameUtils.getSafeName(g, resource));
                Set<Resource> rts = g.getTypes(resource);
                boolean hasArgType = false;
                for (int t = 0; t < argTypes.length; ++t) {
                    if (argTypes[t] == null)
                        continue;
                    if (rts.contains(argTypes[t])) {
                        hasArgType = true;
                        if (DEBUG)
                            System.out.println("IS INSTANCE OF " + args[t]);
                    }
                }
                if (!hasArgType) {
                    if (DEBUG)
                        System.out.println("IS NOT AN INSTANCE OF ANY ARGUMENT TYPE");
                    return false;
                }
            }
            if (DEBUG)
                System.out.println("ALL RESOURCES ARE INSTANCES OF ONE OF: " + Arrays.toString(args));
            return true;
        } else if (ALL_INHERIT.equals(property)) {
            Resource[] argTypes = new Resource[args.length];
            boolean argTypesResolved = false;
            for (int i = 0; i < args.length; i++) {
                try {
                    argTypes[i] = g.getResource(VERSIONS.currentVersion((String) args[i]));
                    argTypesResolved = true;
                } catch (ResourceNotFoundException e) {
                    /* This is a natural situation (database does not contain all resources
                           referred in plugins). No error reporting.
                     */
                }
            }
            if (!argTypesResolved) {
                if (DEBUG)
                    System.out.println("(WW) NO ARGUMENTS RESOLVED INTO RESOURCES!");
                return false;
            }

            for (Resource resource : resources) {
                if (DEBUG)
                    System.out.println("** " + NameUtils.getSafeName(g, resource));
                boolean inheritsArgType = false;
                for (int t = 0; t < argTypes.length; ++t) {
                    if (argTypes[t] == null)
                        continue;
                    if (g.isInheritedFrom(resource, argTypes[t])) {
                        inheritsArgType = true;
                        if (DEBUG)
                            System.out.println("INHERITS " + args[t]);
                    }
                }
                if (!inheritsArgType) {
                    if (DEBUG)
                        System.out.println("DOES NOT INHERIT ANY ARGUMENT TYPE");
                    return false;
                }
            }
            if (DEBUG)
                System.out.println("ALL RESOURCES INHERIT ONE OF: " + Arrays.toString(args));
            return true;
        }
        return false;
    }

}
