/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.selectionview;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.part.Page;
import org.simantics.ui.workbench.IPropertyPage;

/**
 * <p>
 * Subclasses must implement the following methods:
 * <ul>
 *   <li><code>createControl</code> - to create the page's control</li>
 *   <li><code>getControl</code> - to retrieve the page's control</li>
 *   <li><code>getSelection</code> - to retrieve the page's current selection</li>
 *   <li><code>setFocus</code> - implement to accept focus</li>
 *   <li><code>sourceSelectionChanged</code> - puts the incoming ISelection into use in this page</li>
 *   <li><code>sourcePartClosed</code> - cleans up the page controls after a current selection source part has been closed</li>
 * </ul>
 * </p>
 * <p>
 * Subclasses may extend or reimplement the following methods as required:
 * <ul>
 *   <li><code>dispose</code> - extend to provide additional cleanup</li>
 *   <li><code>setActionBars</code> - reimplement to make contributions</li>
 *   <li><code>makeContributions</code> - this method exists to support previous versions</li>
 *   <li><code>setActionBars</code> - this method exists to support previous versions</li>
 *   <li><code>init</code> - extend to provide additional setup</li>
 * </ul>
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public abstract class AbstractPropertyPage extends Page implements IPropertyPage {

    /**
     * Part listener which cleans up this page when the source part is closed.
     * This is hooked only when there is a source part.
     */
    private class PartListener implements IPartListener {
        public void partActivated(IWorkbenchPart part) {
        }

        public void partBroughtToTop(IWorkbenchPart part) {
        }

        public void partClosed(IWorkbenchPart part) {
            if (sourcePart == part) {
                sourcePart = null;
                sourcePartClosed(part);
            }
        }

        public void partDeactivated(IWorkbenchPart part) {
        }

        public void partOpened(IWorkbenchPart part) {
        }
    }

    private final PartListener   partListener = new PartListener();

    protected IWorkbenchPartSite sourceSite;

    protected IWorkbenchPart     sourcePart;

    /**
     * @param sourceSite the workbench part site that produces this page or
     *        <code>null</code> if there is no site, i.e. the page is within a
     *        dialog or a plain shell.
     */
    public AbstractPropertyPage(IWorkbenchPartSite sourceSite) {
        super();
        this.sourceSite = sourceSite;
    }

    @Override
    public void dispose() {
        super.dispose();

        if (sourcePart != null) {
            sourcePart.getSite().getPage().removePartListener(partListener);
            sourcePart = null;
        }

        sourceSite = null;

    }

    @Override
    public void selectionChanged(IWorkbenchPart part, ISelection selection) {
        if (getControl() == null) {
            return;
        }

        if (sourcePart != null) {
            sourcePart.getSite().getPage().removePartListener(partListener);
            sourcePart = null;
        }

        // change the viewer input since the workbench selection has changed.
        sourcePart = part;
        sourceSelectionChanged(selection);

        if (sourcePart != null) {
            IWorkbenchPartSite site = sourcePart.getSite();
            if(site == null) {
                new Exception("null site").printStackTrace();
                return;
            }
            IWorkbenchPage page = site.getPage();
            if(page == null) {
                new Exception("null page").printStackTrace();
                return;
            }
            page.addPartListener(partListener);
        }
    }

    /**
     * @param selection
     */
    protected abstract void sourceSelectionChanged(ISelection selection);

    /**
     * @param part
     */
    protected abstract void sourcePartClosed(IWorkbenchPart part);

}
