package org.simantics.scl.ui.editor2;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ForkJoinPool;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.AnnotationModel;
import org.simantics.scl.compiler.errors.CompilationError;
import org.simantics.scl.compiler.errors.DoesNotExist;
import org.simantics.scl.compiler.errors.ErrorSeverity;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.errors.Failure;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.module.repository.UpdateListener;

public class SCLAnnotationModel extends AnnotationModel {
    private final SCLModuleEditorInput input;
    private final ModuleRepository repository;
    private volatile boolean connected = false;
    
    public SCLAnnotationModel(SCLModuleEditorInput input,
            ModuleRepository repository) {
        this.input = input;
        this.repository = repository;
    }
    
    UpdateListener updateListener = new UpdateListener() {
        @Override
        public void notifyAboutUpdate() {
            if(connected)
                updateAnnotations();
        }
    };
    
    private void updateAnnotations() {
        ForkJoinPool.commonPool().submit(() -> {
            if (!connected)
                return;
            Failable<Module> result = repository.getModule(input.getModuleName(), updateListener);
            if(result instanceof Failure) {
                Failure failure = (Failure)result;
                setAnnotations(Arrays.asList(failure.errors));
            }
            else if(result == DoesNotExist.INSTANCE)
                setAnnotations(Collections.<CompilationError>emptyList());
            else
                setAnnotations(Arrays.asList(result.getResult().getWarnings()));
        });
    }
    
    protected void setAnnotations(List<CompilationError> errors) {
        synchronized(getLockObject()) {
            removeAllAnnotations();
            for(CompilationError error : errors) {
                Annotation annotation = new Annotation(
                        error.severity == ErrorSeverity.ERROR || error.severity == ErrorSeverity.IMPORT_ERROR ?
                                "org.eclipse.ui.workbench.texteditor.error" : //$NON-NLS-1$
                                    "org.eclipse.ui.workbench.texteditor.warning", //$NON-NLS-1$
                                    true, error.description);
                int begin = Locations.beginOf(error.location);
                int end = Locations.endOf(error.location);
                if(begin < 0 || end < begin) {
                    begin = 0;
                    end = 1;
                }
                Position position = new Position(begin, end - begin);
                addAnnotation(annotation, position);
            }
        }
    }

    @Override
    public void connect(IDocument document) {
        super.connect(document);
        connected = true;
        updateAnnotations();
    }
    
    @Override
    public void disconnect(IDocument document) {
        connected = false;
        updateListener.stopListening();
        super.disconnect(document);
    }
}
