package org.simantics.scl.rest;

import java.io.Reader;
import java.io.Writer;
import java.util.concurrent.ConcurrentHashMap;

import org.simantics.scl.compiler.commands.CommandSessionWithModules;
import org.simantics.scl.osgi.SCLOsgi;

public class SCLAPI {

    private static SCLAPI INSTANCE;
    
    private ConcurrentHashMap<String, CommandSessionWithModules> commandSessions;
    
    private SCLAPI() {
        this.commandSessions = new ConcurrentHashMap<>();
    }
    
    public static SCLAPI getInstance() {
        if (INSTANCE == null) {
            synchronized (SCLAPI.class) {
                if (INSTANCE == null) {
                    INSTANCE = new SCLAPI();
                }
            }
        }
        return INSTANCE;
    }

    public CommandSessionWithModules getOrCreateCommandSession(String sessionId) {
        return commandSessions.computeIfAbsent(sessionId, key -> new CommandSessionWithModules(SCLOsgi.MODULE_REPOSITORY));
    }

    public void execute(String sessionId, Reader reader, Writer writer) {
        CommandSessionWithModules session = commandSessions.get(sessionId);
        if (session == null)
            throw new IllegalArgumentException("CommandSession for sessionId " + sessionId + " does not exist!");
        session.runCommands(reader, writer);
    }

    public void deleteCommandSession(String sessionId) {
        commandSessions.computeIfPresent(sessionId, (key, session) -> {
            // session could be flushed or closed here to release possible resources?
            return null;
        });
    }

    public Object variableValue(String sessionId, String variableName) {
        CommandSessionWithModules session = commandSessions.get(sessionId);
        if (session == null)
            throw new IllegalArgumentException("CommandSession for sessionId " + sessionId + " does not exist!");
        return session.getCommandSession().getVariableValue(variableName);
    }

    public String putModule(String sessionId, String moduleName, String moduleText) {
        CommandSessionWithModules session = commandSessions.get(sessionId);
        if (session == null)
            throw new IllegalArgumentException("CommandSession for sessionId " + sessionId + " does not exist!");
        return session.putModule(moduleName, moduleText);
    }
}
