package org.simantics.scl.db;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.QueryIndexUtils;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UsedSCLExpressionsRequest extends UniqueRead<Collection<SCLExpressionTableEntry>> {

    private static final Logger LOGGER = LoggerFactory.getLogger(UsedSCLExpressionsRequest.class);

    private static final String structuralResourceComponentURI = "http://www.simantics.org/Structural-1.2/Component";

    @Override
    public Collection<SCLExpressionTableEntry> perform(ReadGraph graph) throws DatabaseException {
        Collection<SCLExpressionTableEntry> result = new ArrayList<>();
        Layer0 L0 = Layer0.getInstance(graph);
        
        Set<Resource> indexRoots = new TreeSet<Resource>();
        for(Resource ontology : Layer0Utils.listOntologies(graph)) {
            if (graph.isInstanceOf(ontology, L0.SharedOntology)) {
                indexRoots.add(ontology);
            }
        }
        for(Resource child : graph.getObjects(Simantics.getProjectResource(), L0.ConsistsOf)) {
            if (graph.isInstanceOf(child, L0.IndexRoot)) {
                indexRoots.add(child);
            }
        }

        Resource componentResource = graph.getPossibleResource(structuralResourceComponentURI);
        if (componentResource != null) {
            for (Resource ontology : indexRoots) {
                List<Resource> components = QueryIndexUtils.searchByTypeShallow(graph, ontology, componentResource);
                for (Resource component : components) {
                    for (Statement propertyStatement : graph.getStatements(component, L0.HasProperty)) {
                        if (graph.isInstanceOf(propertyStatement.getObject(), L0.SCLValue)) {
                            Resource sclValue = propertyStatement.getObject();
                            String expression = graph.getPossibleRelatedValue2(sclValue, L0.SCLValue_expression);
                            Resource source = graph.getPossibleObject(sclValue, L0.PropertyOf);
                            if (source != null) {
                                String uri = graph.getPossibleURI(source);
                                String pred = graph.getRelatedValue2(propertyStatement.getPredicate(), L0.HasName);
    
                                if (uri != null) {
                                    result.add(new SCLExpressionTableEntry(expression, uri + "#" + pred, source));
                                }
                            }
                        }
                    }
                }
            }
        } else {
            LOGGER.info("{} is not available for finding expressions in component properties", structuralResourceComponentURI);
        }
        return result;
    }

    public static Collection<SCLExpressionTableEntry> execute() {
        try {
            return Simantics.getSession().syncRequest(new UsedSCLExpressionsRequest(), TransientCacheListener.instance());
        } catch (DatabaseException e) {
            e.printStackTrace();
            return Collections.emptyList();
        }
    }

}
