/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d;

import java.awt.Composite;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.events.IEventHandler;

/**
 * 
 * 
 * @author Tuukka Lehtonen
 */
public interface IG2DNode extends INode, IEventHandler {

    /**
     * Set the z-index, i.e. the rendering order number for this 2D node.
     * 2D nodes are sorted in ascending z-order for rendering.
     * 
     * @param z
     */
    @SyncField("z")
    public void setZIndex(int z);

    /**
     * @return the rendering order of this 2D node.
     */
    public int getZIndex();

    /**
     * @return <code>false</code> if this node will not render anything if
     *         {@link #render(Graphics2D)} was invoked. Serves as a means to
     *         optimize rendering. Should always return <code>true</code> by
     *         default.
     */
    public boolean validate();

    /**
     * Perform the actual rendering. Node state changes must be avoided in
     * this method. Implement node state changes to refresh method instead. 
     * This method should not change the transform of the g2d
     * instance. You should call g2d.create or copy the original transform and
     * put it back at the end of this method according to what is specified in
     * {@link Graphics2D#setTransform(AffineTransform)}.
     * 
     * This method is always invoked so that the local transform of this
     * {@link IG2DNode} has not been applied to the Graphics2D rendering
     * context. It is this node's responsibility to apply its own transform
     * during rendering if necessary.
     * 
     * The method must also make sure that it returns the {@link Composite} of
     * g2d to its original state using (see {@link Graphics2D#setComposite(Composite)).
     * 
     * @param g2d
     * 
     * @see #setTransform(AffineTransform)
     * @see #getTransform()
     */
    public void render(Graphics2D g2d);
    
    /**
     * Refresh the node state.  
     */
    public void refresh();
    
    /**
     * Visit the IG2DNode substructure of this node using the provided visitor.
     * 
     * @param visitor the visitor to use
     * @since 1.29.0
     */
    public void accept(IG2DNodeVisitor visitor);

    /**
     * Mark the scene graph to be repainted in its current rendering context (UI
     * component) as soon as is appropriate for the backend system.
     */
    public void repaint();

    // Bounds and transformation

    /**
     * @return affine transformation set for this node to transform it into its
     *         parent's coordinate space
     */
    public AffineTransform getTransform();

    /**
     * Set an affine transformation for this node that transforms its local
     * coordinates into its parent's coordinate space
     * 
     * @param transform local &rarr; parent transform
     */
    public void setTransform(AffineTransform transform);

    /**
     * Return bounds transformed with local transformation
     * 
     * @return transformed bounds
     */
    public Rectangle2D getBounds();

    /**
     * Return bounds in local coordinates. May expose internal state of the
     * node. The client is not allowed to modify the returned value.
     * 
     * @return bounds
     */
    public Rectangle2D getBoundsInLocal();

    public Rectangle2D getBoundsInLocal(boolean ignoreNulls);

    // Helper methods for bounds checking

    public boolean contains(Point2D point);

    public boolean intersects(Rectangle2D rect);

    public Point2D localToParent(Point2D point);

    public Rectangle2D localToParent(Rectangle2D rect);

    public Point2D parentToLocal(Point2D point);

    public Rectangle2D parentToLocal(Rectangle2D rect);

    public Point2D localToControl(Point2D point);

    public Rectangle2D localToControl(Rectangle2D rect);

    public Point2D controlToLocal(Point2D point);

    public Rectangle2D controlToLocal(Rectangle2D rect);

    /**
     * @return root node of the 2D scene graph or <code>null</code> if this node is
     *         not part of a properly rooted scene graph hierarchy
     */
    public G2DSceneGraph getRootNode2D();

    /**
     * @return true if this node has focus in the owner scene graph
     */
    public boolean hasFocus();

    /**
     * @return the node that has focus in the owner scene graph
     */
    public IG2DNode getFocusNode();

    /**
     * @param node the node to have focus in the owner scene graph. The
     *        specified node must be a part of the same scene graph.
     */
    public void setFocusNode(IG2DNode node);

}
