/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.profile.common;

import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.NodeException;
import org.simantics.scenegraph.ParentNode;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.profile.EvaluationContext;
import org.simantics.scenegraph.profile.Observer;
import org.simantics.scenegraph.profile.Style;
import org.simantics.scenegraph.utils.InitValueSupport;
import org.simantics.scenegraph.utils.NodeUtil;

/**
 * Utilities for handling scene graph nodes in diagram profile styles.
 * 
 * @see Style
 */
public class ProfileVariables {

    public static void init(INode node, Observer observer) {
        if(node instanceof InitValueSupport) {
            ((InitValueSupport)node).initValues();
        }
    }
    
    /**
     * @param node
     * @param id
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <N extends INode> N browseChild(INode node, String id) {

        // There was an empty string meaning that input should be returned
        if (id.isEmpty())
            return (N)node;

        String[] parts = id.split("\\.");
        for (String part : parts) {
            if ("*".equals(part)) {
                node = NodeUtil.getFirstChild(node);
            } else {
                node = NodeUtil.getChildById(node, part);
            }
            if (node == null)
                return null;
        }

        return (N)node;

    }

    /*
     * Sets named property for named node starting from top level. The path identifies a node from named child hierarchy (separated by '.').
     */
    public static void claimNodeProperty(INode node, String property, Object value, EvaluationContext evaluationContext) {
        
        if(node == null) {
            evaluationContext.update();
            return;
        }
        
//        for(String property : concat.keySet()) {
            //System.out.println("sync: "+property+" | "+concat.get(property));
            // TODO: property names containing dots "." should be splitted and treated as hierarchical properties 
            if(property.contains(".")) {
                String t[] = property.split("\\.");
                if(t.length == 2) { // FIXME: add support for deeper hierarchy
                    String child_name = t[0];
                    String property_name = t[1];
                    if(node instanceof G2DParentNode) {
                        INode child = NodeUtil.findChildById((G2DParentNode)node, child_name);
                        if(child != null)
                            NodeUtil.setPropertyIfSupported(property_name, value, child);
                    }
                }
            } else {
                NodeUtil.setPropertyIfSupported(property, value, node);
            }
//        }
        
    }

    /*
     * Ensures that the element contains a named child node in given path with
     * given class. All nodes in the path are expected to exist.
     */
    public static <N extends INode> N claimChild(INode node, String path, String name, Class<N> clazz, Observer observer) {

        INode child = browseChild(node, path);
        if(child == null) {
            observer.exception(new NullPointerException("Scenegraph child node was not found: " + path));
            observer.update();
            return null;
        }

        INode existing = NodeUtil.getChildById(child, name);
        if (existing == null) {
            if (child instanceof ParentNode<?>) {
                existing = ((ParentNode<?>) child).addNode(name, clazz);
            } else {
                observer.exception(new NodeException("Cannot claim child node for non-parent-node " + child));
                return null;
            }
        } else if (!clazz.isInstance(existing)) {
            observer.exception(new ClassCastException("Class of existing child node (" + existing.getClass() + ") with path=" + path + " and name=" + name + " is invalid, requested " + clazz));
            return null;
        }
        return clazz.cast(existing);
    }

    /**
     * @param node
     * @param path
     * @param name
     */
    public static void denyChild(INode node, String path, String name) {
        INode child = browseChild(node, path);
        if (child == null)
            return;

        if (child instanceof ParentNode<?>) {
            ((ParentNode<?>) child).removeNode(name);
        }
    }

    public static void denyChild(INode node, String name) {
    	denyChild(node, "", name);
    }

    public static void denyChildren(INode node, String prefix) {
        if (node instanceof ParentNode<?>) {
            ParentNode<?> parent = (ParentNode<?>) node;

            for (String childId : NodeUtil.filterDirectChildIds(parent, prefix))
                parent.removeNode(childId);
        }
    }

}
