/*******************************************************************************
 * Copyright (c) 2012, 2013 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.structural;

import java.util.ArrayList;
import java.util.List;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;

/**
 * An object representing structural Resource.
 * 
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class StructuralResource  {

	private Resource resource;
	private List<Resource> context = new ArrayList<Resource>(1);
	
	private Resource typeResource = null;
	
	public StructuralResource(Resource resource) {
		assert(resource != null);
		this.resource = resource;
	}
	public StructuralResource(ReadGraph g, Resource resource, Resource context) throws DatabaseException {
		assert(resource != null);
		this.resource = resource;
		this.context.add(context);
		resolveType(g);
	}
	
	public StructuralResource(ReadGraph g, Resource resource, Resource... context) throws DatabaseException {
		assert(resource != null);
		this.resource = resource;
		for (Resource r : context)
			this.context.add(r);
		resolveType(g);
	}
	
	public StructuralResource(ReadGraph g, Resource resource, List<Resource> context) throws DatabaseException {
		assert(resource != null);
		this.resource = resource;
		for (Resource r : context)
			this.context.add(r);
		resolveType(g); 
	}
	public StructuralResource(ReadGraph g, Resource resource, List<Resource> context, Resource context2) throws DatabaseException {
		assert(resource != null);
		this.resource = resource;
		for (Resource r : context)
			this.context.add(r);
		this.context.add(context2);
		resolveType(g);
	}
	
	private void resolveType(ReadGraph g) throws DatabaseException {
		if (this.context.contains(resource)) {
			Layer0 l0 = Layer0.getInstance(g);
			typeResource = g.getSingleObject(resource, l0.InstanceOf);
		} 
	}
	
	
	/**
	 * The Resource in the DB.
	 * @return
	 */
	public Resource getResource() {
		return resource;
	}
	
	/**
	 * Context in which this resource is accessed. Each context resource represents a structural model instance. 
	 * @return
	 */
	public List<Resource> getContext() {
		return context;
	}
	
	/**
	 * If the resource is structural model instance, this returns the type Resource. Otherwise returns null.
	 * @return
	 */
	public Resource getTypeResource() {
		return typeResource;
	}
	
	/**
	 * Returns true, if the resource is structural, 
	 * @return
	 */
	public boolean isStructural() {
		return context.size() > 0;
	}
	
	/**
	 * Returns true is the Resource is root of Structural Model instance.
	 * In this case the resource instance is editable.
	 * 
	 * @return
	 */
	public boolean isStructuralRoot() {
		return (context.size() == 1 && context.get(0).equals(resource));
	}
	
	/**
	 * Returns true,  the resource is structural model instance.
	 * @return
	 */
	public boolean isStructuralInstance() {
		return typeResource != null;
	}
	
	@Override
	public int hashCode() {
		int hashCode = resource.hashCode();
		for (Resource ctx : context)
			hashCode += ctx.hashCode();
		return hashCode;
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == this)
			return true;
		if (obj == null)
			return false;
		if (obj.getClass() != getClass())
			return false;
		StructuralResource other = (StructuralResource)obj;
		if (!resource.equals(other.resource))
			return false;
		if (context.size() != other.context.size())
			return false;
		for (int i = 0; i < context.size(); i++) {
			if (!context.get(i).equals(other.context.get(i)))
				return false;
		}
		return true;
	}
	
	@Override
	public String toString() {
		String s = "Res: " + resource + " Context:";
		for (Resource ctx : context) 
			s+= " "+ ctx;
		if (typeResource != null)
			s+= " Type: " + typeResource;
		return s;
	}

}
