/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.mapping.constraint.instructions;

import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.set.hash.TIntHashSet;

import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.mapping.IContextualModification;

public abstract class CombinedInstruction implements IInstruction {
	IInstruction[] instructions;
	
	public CombinedInstruction(IInstruction... instructions) {
		this.instructions = instructions;
	}	
	
	class CompositeClaim implements IContextualModification {

		IContextualModification modification;
		int instructionId;
		
		public CompositeClaim(IContextualModification modification, int instructionId) {
			this.modification = modification;
			this.instructionId = instructionId;
		}

		@Override
		public void perform(WriteGraph g, Object[] bindings) throws DatabaseException {
			modification.perform(g, bindings);
			for(int i=instructionId+1;i<instructions.length;++i)
				instructions[i].doClaim(g, bindings);
		}
		
	}
	
	@Override
	public IContextualModification claim(ReadGraph g, Object[] bindings) throws DatabaseException {
		for(int instructionId=0;instructionId<instructions.length;++instructionId) {
			IContextualModification modification = instructions[instructionId].claim(g, bindings);
			if(modification != null) {
				if(instructionId==instructions.length-1)
					return modification;
				else 
					return new CompositeClaim(modification, instructionId);
			}
		}
		return null;
	}

	class CompositeDeny implements IContextualModification {

		IContextualModification modification;
		int instructionId;
		
		public CompositeDeny(IContextualModification modification, int instructionId) {
			this.modification = modification;
			this.instructionId = instructionId;
		}

		@Override
		public void perform(WriteGraph g, Object[] bindings) throws DatabaseException {
			modification.perform(g, bindings);
			for(int i=instructionId+1;i<instructions.length;++i)
				instructions[i].doDeny(g, bindings);
		}
		
	}

	@Override
	public IContextualModification deny(ReadGraph g, Object[] bindings) throws DatabaseException {
		for(int instructionId=0;instructionId<instructions.length;++instructionId) {
			IContextualModification modification = instructions[instructionId].deny(g, bindings);
			if(modification != null) {
				if(instructionId==instructions.length-1)
					return modification;
				else 
					return new CompositeDeny(modification, instructionId);
			}
		}
		return null;
	}

	@Override
	public void doClaim(WriteGraph g, Object[] bindings) throws DatabaseException {
		for(int i=0;i<instructions.length;++i)
			instructions[i].doClaim(g, bindings);		
	}

	@Override
	public void doDeny(WriteGraph g, Object[] bindings) throws DatabaseException {
		for(int i=0;i<instructions.length;++i)
			instructions[i].doDeny(g, bindings);					
	}

	@Override
	public void collectVariables(TIntHashSet reads, TIntHashSet writes) {
		for(IInstruction inst : instructions)
			inst.collectVariables(reads, writes);
	}	
	
	@Override
	public void mapVariables(TIntIntHashMap map) {
		for(IInstruction inst : instructions)
			inst.mapVariables(map);
	}

}
