/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.layer0.utils.direct;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.Stack;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.uri.UnescapedChildMapOfResource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.procedure.MultiProcedure;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.layer0.Layer0;


public final class GraphUtils {

    public static Resource createScalarString(WriteGraph graph, String string) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        Resource ret = graph.newResource();
        graph.claim(ret, b.InstanceOf, null, b.String);
        graph.claimValue(ret, string);
        return ret;
    }

    public static Resource createScalarInteger(WriteGraph graph, int value) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        Resource ret = graph.newResource();
        graph.claim(ret, b.InstanceOf, null, b.Integer);
        graph.claimValue(ret, value);
        return ret;
    }

    public static Resource createScalarLong(WriteGraph graph, long value) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        Resource ret = graph.newResource();
        graph.claim(ret, b.InstanceOf, null, b.Long);
        graph.claimValue(ret, value);
        return ret;
    }

    public static Resource createScalarFloat(WriteGraph graph, float value) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        Resource ret = graph.newResource();
        graph.claim(ret, b.InstanceOf, null, b.Float);
        graph.claimValue(ret, value);
        return ret;
    }

    public static Resource createScalarBoolean(WriteGraph graph, boolean value) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        Resource ret = graph.newResource();
        graph.claim(ret, b.InstanceOf, null, b.Boolean);
        graph.claimValue(ret, value);
        return ret;
    }

    public static Resource createScalarDouble(WriteGraph graph, double value) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        Resource ret = graph.newResource();
        graph.claim(ret, b.InstanceOf, null, b.Double);
        graph.claimValue(ret, value);
        return ret;
    }

    public static void addRelatedScalarString(WriteGraph graph, Resource resource, Resource relation, String string) throws DatabaseException {
        graph.claim(resource, relation, createScalarString(graph, string));
    }

    public static void addRelatedScalarInteger(WriteGraph graph, Resource resource, Resource relation, int value) throws DatabaseException {
        graph.claim(resource, relation, createScalarInteger(graph, value));
    }

    public static Resource createDoubleArray(WriteGraph graph, double[] array, Resource type) throws DatabaseException {
        Resource ret = graph.newResource();
    	Layer0 b = Layer0.getInstance(graph);
        graph.claim(ret, b.InstanceOf, null, type);
        graph.claimValue(ret, array);
        return ret;
    }

    public static Resource createDoubleArray(WriteGraph graph, double[] array) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        return createDoubleArray(graph, array, b.DoubleArray);
    }

    public static Resource createIntegerArray(WriteGraph graph, int[] array, Resource type) throws DatabaseException {
        Resource ret = graph.newResource();
        Layer0 b = Layer0.getInstance(graph);
        graph.claim(ret, b.InstanceOf, null, type);
        graph.claimValue(ret, array);
        return ret;
    }

    public static Resource createIntegerArray(WriteGraph graph, int[] array) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        return createIntegerArray(graph, array, b.IntegerArray);
    }

    public static Resource createLongArray(WriteGraph graph, long[] array, Resource type) throws DatabaseException {
        Resource ret = graph.newResource();
        Layer0 b = Layer0.getInstance(graph);
        graph.claim(ret, b.InstanceOf, null, type);
        graph.claimValue(ret, array);
        return ret;
    }

    public static Resource createLongArray(WriteGraph graph, long[] array) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        return createLongArray(graph, array, b.LongArray);
    }

    public static Resource createFloatArray(WriteGraph graph, float[] array, Resource type) throws DatabaseException {
        Resource ret = graph.newResource();
        Layer0 b = Layer0.getInstance(graph);
        graph.claim(ret, b.InstanceOf, null, type);
        graph.claimValue(ret, array);
        return ret;
    }

    public static Resource createFloatArray(WriteGraph graph, float[] array) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        return createFloatArray(graph, array, b.FloatArray);
    }

    public static Resource createBooleanArray(WriteGraph graph, boolean[] array, Resource type) throws DatabaseException {
        Resource ret = graph.newResource();
        Layer0 b = Layer0.getInstance(graph);
        graph.claim(ret, b.InstanceOf, null, type);
        graph.claimValue(ret, array);
        return ret;
    }

    public static Resource createBooleanArray(WriteGraph graph, boolean[] array) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        return createBooleanArray(graph, array, b.BooleanArray);
    }

    public static Resource createStringArray(WriteGraph graph, String[] array, Resource type) throws DatabaseException {
        Resource ret = graph.newResource();
        Layer0 b = Layer0.getInstance(graph);
        graph.claim(ret, b.InstanceOf, null, type);
        graph.claimValue(ret, array);
        return ret;
    }

    public static Resource createStringArray(WriteGraph graph, String[] array) throws DatabaseException {
        Layer0 b = Layer0.getInstance(graph);
        return createStringArray(graph, array, b.StringArray);
    }

    public interface ResourceTester {
        boolean test(ReadGraph graph, Resource resource) throws DatabaseException ;
    }

    public static class AcceptAll implements ResourceTester {

        @Override
        public boolean test(ReadGraph graph, Resource resource) throws DatabaseException {
            return true;
        }

    }

    public static class InstanceOf implements ResourceTester {

        private final Resource type;

        public InstanceOf(Resource type) {
            this.type = type;
        }

        @Override
        public boolean test(ReadGraph graph, Resource resource) throws DatabaseException {
            return graph.isInstanceOf(resource, type);
        }

    }

    public static class Inherits implements ResourceTester {

        private final Resource superType;

        public Inherits(Resource superType) {
            this.superType = superType;
        }

        @Override
        public boolean test(ReadGraph graph, Resource resource) throws DatabaseException {
            return graph.isInheritedFrom(resource, superType);
        }

    }

    public static void findResources(ReadGraph graph, Collection<Resource> roots, Resource relation, ResourceTester tester, MultiProcedure<Resource> procedure) throws DatabaseException {

        Set<Resource> visited = new HashSet<Resource>();
        Stack<Resource> process = new Stack<Resource>();

        process.addAll(roots);
        visited.addAll(roots);

        while(!process.isEmpty()) {
            Resource cur = process.pop();
            for(Resource r : graph.getObjects(cur, relation))
                if(!visited.contains(r)) {
                    visited.add(r);
                    if(tester.test(graph, r))
                        procedure.execute(r);
                    process.add(r);
                }
        }

        procedure.finished();

    }


    public static void findResources(ReadGraph graph, Collection<Resource> roots, Resource relation, ResourceTester tester, SyncMultiProcedure<Resource> procedure) throws DatabaseException {

        Set<Resource> visited = new HashSet<Resource>();
        Stack<Resource> process = new Stack<Resource>();

        process.addAll(roots);
        visited.addAll(roots);

        while(!process.isEmpty()) {
            Resource cur = process.pop();
            for(Resource r : graph.getObjects(cur, relation))
                if(!visited.contains(r)) {
                    visited.add(r);
                    if(tester.test(graph, r))
                        procedure.execute(graph, r);
                    process.add(r);
                }
        }

        procedure.finished(graph);

    }

    /**
     * @deprecated use {@link NameUtils#getSafeName(ReadGraph, Resource)}
     */
    public static String getReadableName(ReadGraph graph, Resource resource) throws ValidationException, ServiceException {
        return NameUtils.getSafeName(graph, resource);
    }

    /**
     * @deprecated use {@link NameUtils#findReservedNames(ReadGraph, String, Resource, Resource, Set)}
     */
    public static Set<String> findReservedNames(ReadGraph g, String proposition, Resource container, Resource consistRelation, Set<String> result) throws DatabaseException {
        return NameUtils.findReservedNames(g, proposition, container, consistRelation, result);
    }

    /**
     * @deprecated use {@link NameUtils#findReservedNames(ReadGraph, String, Resource, Resource)
     */
    public static Set<String> findReservedNames(ReadGraph g, String proposition, Resource container, Resource consistRelation) throws DatabaseException {
        return NameUtils.findReservedNames(g, proposition, container, consistRelation);
    }

    /**
     * @deprecated use {@link NameUtils#findFreshName(ReadGraph, String, Resource)}
     */
    public static String findFreshName(ReadGraph g, String proposition, Resource container) throws DatabaseException {
        return NameUtils.findFreshName(g, proposition, container);
    }

    /**
     * @deprecated use {@link NameUtils#findFreshEscapedName(ReadGraph, String, Resource)}
     */
    public static String findFreshEscapedName(ReadGraph g, String proposition, Resource container) throws DatabaseException {
        return NameUtils.findFreshEscapedName(g, proposition, container);
    }

    /**
     * @deprecated use {@link NameUtils#findFreshName(ReadGraph, String, Resource, Resource)}
     */
    public static String findFreshName(ReadGraph g, String proposition, Resource container, Resource consistRelation) throws DatabaseException {
        return NameUtils.findFreshName(g, proposition, container, consistRelation);
    }

    /**
     * @deprecated use {@link NameUtils#findFreshEscapedName(ReadGraph, String, Resource, Resource)}
     */
    public static String findFreshEscapedName(ReadGraph g, String proposition, Resource container, Resource consistRelation) throws DatabaseException {
        return NameUtils.findFreshEscapedName(g, proposition, container, consistRelation);
    }

    /**
     * @deprecated use {@link NameUtils#findFreshName(ReadGraph, String, Resource, Resource, String)}
     */
    public static String findFreshName(ReadGraph g, String proposition, Resource container, Resource consistRelation, String nameFormat) throws DatabaseException {
        return NameUtils.findFreshName(g, proposition, container, consistRelation, nameFormat);
    }

    /**
     * @deprecated use {@link NameUtils#findFreshInstanceName(ReadGraph, Resource, Resource)}
     */
    public static String findFreshInstanceName(ReadGraph g, Resource type, Resource container) throws DatabaseException {
        return NameUtils.findFreshInstanceName(g, type, container);
    }

    /**
     * @deprecated use {@link NameUtils#findFreshInstanceName(ReadGraph, Resource, Resource, Resource)}
     */
    public static String findFreshInstanceName(ReadGraph g, Resource type, Resource container, Resource relation) throws DatabaseException {
        return NameUtils.findFreshInstanceName(g, type, container, relation);
    }

    public static Resource create(WriteGraph g, Resource ... predicateObjectPairs) throws DatabaseException {
        assert(predicateObjectPairs.length % 2 == 0);
        Resource resource = g.newResource();
        for(int i=0;i<predicateObjectPairs.length;i+=2)
            g.claim(resource, predicateObjectPairs[i], predicateObjectPairs[i+1]);
        return resource;
    }

    private static Resource getObjectOrCreatePrimitiveValue(WriteGraph g, Object object) throws DatabaseException {
        if (object == null)
            throw new NullPointerException("null object");

        Class<?> clazz = object.getClass();

        if (object instanceof Resource)
            return (Resource) object;

        if (String.class.equals(clazz))
            return createScalarString(g, (String) object);
        else if (String[].class.equals(clazz))
            return createStringArray(g, (String[]) object);
        else if (Integer.class.equals(clazz))
            return createScalarInteger(g, (Integer) object);
        else if (int[].class.equals(clazz))
            return createIntegerArray(g, (int[]) object);
        else if (Long.class.equals(clazz))
            return createScalarLong(g, (Long) object);
        else if (long[].class.equals(clazz))
            return createLongArray(g, (long[]) object);
        else if (Float.class.equals(clazz))
            return createScalarFloat(g, (Float) object);
        else if (float[].class.equals(clazz))
            return createFloatArray(g, (float[]) object);
        else if (Double.class.equals(clazz))
            return createScalarDouble(g, (Double) object);
        else if (double[].class.equals(clazz))
            return createDoubleArray(g, (double[]) object);
        else if (Boolean.class.equals(clazz))
            return createScalarBoolean(g, (Boolean) object);
        else if (boolean[].class.equals(clazz))
            return createBooleanArray(g, (boolean[]) object);

        throw new UnsupportedOperationException("unsupported object type: " + object);
    }

    public static Resource create(WriteGraph g, Object ... predicateObjectPairs) throws DatabaseException {
        assert(predicateObjectPairs.length % 2 == 0);
        Resource resource = g.newResource();
        for(int i=0;i<predicateObjectPairs.length;i+=2) {
            Resource predicate = (Resource)predicateObjectPairs[i];
            Object _object = predicateObjectPairs[i+1];
            Resource object = getObjectOrCreatePrimitiveValue(g, _object);
            g.claim(resource, predicate, object);
        }
        return resource;
    }

    public static Resource create2(WriteGraph g, Resource type, Object ... predicateObjectPairs) throws DatabaseException {
        assert(predicateObjectPairs.length % 2 == 0);
        assert(type != null);
        Resource resource = g.newResource();
    	Layer0 b = Layer0.getInstance(g);
        g.claim(resource, b.InstanceOf, null, type);
        for(int i=0;i<predicateObjectPairs.length;i+=2) {
            Resource predicate = (Resource)predicateObjectPairs[i];
            Object _object = predicateObjectPairs[i+1];
            Resource object = getObjectOrCreatePrimitiveValue(g, _object);
            g.claim(resource, predicate, object);
        }
        return resource;
    }

    /**
     * @deprecated use {@link NameUtils#toString(ReadGraph, Statement)}
     */
    public static String toString(ReadGraph g, Statement stm) throws DatabaseException {
        return NameUtils.toString(g, stm);
    }

//    public static Resource getPossiblePath(ReadGraph graph, Resource resource, Resource ... path) throws DatabaseException {
//
//        for(Resource p : path) {
//            resource = graph.getPossibleObject(resource, p);
//            if(resource == null) return null;
//        }
//        return resource;
//
//    }
    
    public static Resource getPossibleChild(ReadGraph g, Resource parent, String name) throws DatabaseException {
        return g.syncRequest(new UnescapedChildMapOfResource(parent)).get(name);
    }
}
