/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.issues.common;

import gnu.trove.map.hash.THashMap;
import gnu.trove.set.hash.THashSet;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.QuaternaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.issues.Severity;
import org.simantics.issues.common.preferences.IssuePrefs;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;

/**
 * @author Tuukka Lehtonen
 * @see CountModelIssuesBySeverity
 */
public class ListModelIssuesBySeverity extends QuaternaryRead<Resource, Boolean, Boolean, Severity, Map<Severity, List<Resource>>> {

    public ListModelIssuesBySeverity(Resource model, boolean onlyUnresolved, boolean respectVisibility, Severity minSeverity) {
        super(model, onlyUnresolved, respectVisibility, minSeverity);
    }

    @SuppressWarnings("unchecked")
    @Override
    public Map<Severity, List<Resource>> perform(ReadGraph graph) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        IssueResource ISSUE = IssueResource.getInstance(graph);

        Severity[] keys = Severity.values();
        @SuppressWarnings("rawtypes")
        List[] values = new List[keys.length];

        Resource project = Simantics.getProjectResource();
        boolean showHidden = false;
        boolean showNormal = true;
        boolean showUser = true;
        if (parameter3 && project != null) {
            showHidden = IssuePrefs.showHiddenIssues(graph, project);
            showNormal = IssuePrefs.showNormalIssues(graph, project);
            showUser = IssuePrefs.showUserIssues(graph, project);
        }

        Instances issueIndex = graph.getPossibleAdapter(ISSUE.Issue, Instances.class);
        Collection<Resource> modelIssues = graph.syncRequest(new ObjectsWithType(parameter, L0.ConsistsOf, ISSUE.Issue));
        Collection<Resource> indexedIssues = issueIndex != null ? issueIndex.find(graph, parameter) : Collections.<Resource>emptyList();
        Collection<Resource> issues = !indexedIssues.isEmpty() ? new THashSet<Resource>(modelIssues.size() + indexedIssues.size()) : modelIssues;
        if (!indexedIssues.isEmpty()) {
            issues.addAll(modelIssues);
            issues.addAll(indexedIssues);
        }

        final boolean ignoreResolved = parameter2;
        for (Resource issue : issues) {
            if (ignoreResolved && graph.hasStatement(issue, ISSUE.Resolved))
                continue;
            boolean hidden = graph.hasStatement(issue, ISSUE.Hidden);
            boolean user = graph.hasStatement(issue, ISSUE.UserIssue);
            boolean normal = !hidden && !user;
            if (!showHidden && hidden)
                continue;
            if (!showUser && user)
                continue;
            if (!showNormal && normal)
                continue;

            Severity s = IssueUtils.toSeverity(ISSUE, graph.getPossibleObject(issue, ISSUE.Issue_HasSeverity));
            if (null != s && s.compareTo(parameter4) <= 0) {
                int index = s.ordinal();
                @SuppressWarnings("rawtypes")
                List list = values[index];
                if (list == null)
                    list = values[index] = new ArrayList<Resource>();
                list.add(issue);
            }
        }

        THashMap<Severity, List<Resource>> result = new THashMap<Severity, List<Resource>>(values.length);
        for (int i = 0; i < values.length; ++i)
            if (values[i] != null)
                result.put(keys[i], values[i]);
        return result;
    }

}
