/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.history.util.subscription;

import java.util.Collection;
import java.util.Iterator;

import org.simantics.databoard.annotations.Identifier;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.util.Bean;

/**
 * (Utility Class) Item format for HistoryManager, Collector, and meta-data for Simantics subscription configurations.
 * 
 * The items objects the HistoryManager and Collector uses are complete meta-data
 * descriptions. Different formats are supported and the content is written to disc
 * completely as is. 
 * 
 * The "id" is composed with the following aggregation: [subscriptionId] [variableId] [formatId].
 * 
 * @author toni.kalajainen
 */
public class SubscriptionItem extends Bean {

	/**
	 * Create HistoryItem descriptions for collecting one variable with multiple
	 * sampling formats.
	 * 
	 * @param item
	 *            the item to use as a template for the created sampled items
	 * @param groupItemId
	 *            the group item id to use with
	 *            {@link #composeItemName(String, String, String)}
	 * @param groupId
	 *            the group id to use with
	 *            {@link #composeItemName(String, String, String)}
	 * @param formats
	 *            the sampling formats to create
	 * @return an array of history items
	 */
	public static SubscriptionItem[] createItems(
			SubscriptionItem item,
			String groupItemId,
			String groupId,
			Collection<SamplingFormat> formats) {
		Iterator<SamplingFormat> itr = formats.iterator();
		SubscriptionItem[] items = new SubscriptionItem[ formats.size() ];
		for (int i=0; i<formats.size(); i++) {
			items[i] = createItem( item, groupItemId, groupId, itr.next() );
		}
		return items;
	}

	/**
	 * Create HistoryItem descriptions for collecting one variable with multiple
	 * sampling formats.
	 * 
	 * @param variableId
	 * @param subscriptionId
	 * @param formats
	 * @return an array of history items
	 */
	public static SubscriptionItem[] createItems(
			String variableId,
			String subscriptionId,
			SamplingFormat...formats
				) {
			SubscriptionItem[] items = new SubscriptionItem[ formats.length ];
			for (int i=0; i<formats.length; i++) {
				items[i] = createItem( variableId, subscriptionId, formats[i] );
			}
			return items;
		}
		
	/**
	 * Create HistoryItem descriptions for collecting one variable with multiple
	 * sampling formats.
	 * 
	 * @param groupItemId
	 * @param groupId
	 * @param formats
	 * @return an array of history items
	 */
	public static SubscriptionItem[] createItems(
			String groupItemId,
			String groupId,
			Collection<SamplingFormat> formats) {
			Iterator<SamplingFormat> itr = formats.iterator();
			SubscriptionItem[] items = new SubscriptionItem[ formats.size() ];
			for (int i=0; i<formats.size(); i++) {
				items[i] = createItem( groupItemId, groupId, itr.next() );
			}
			return items;
		}

	/**
	 * Create HistoryItem description for collecting one variable with one
	 * sampling format.
	 * 
	 * @param item
	 *            the subscription item to clone values from
	 * @param groupItemId
	 *            the group item id to use with
	 *            {@link #composeItemName(String, String, String)}
	 * @param groupId
	 *            the group id to use with
	 *            {@link #composeItemName(String, String, String)}
	 * @param format
	 * @return HistoryItem description.
	 */
	public static SubscriptionItem createItem(
			SubscriptionItem item,
			String groupItemId,
			String groupId,
			SamplingFormat format
			) {
		SubscriptionItem hi = new SubscriptionItem();
		hi.variableId = item.variableId;
		hi.groupItemId = item.groupItemId;
		hi.id = composeItemName(groupId, groupItemId, format.formatId);
		hi.deadband = format.deadband;
		hi.interval = format.interval;
		hi.gain = item.gain;
		hi.bias = item.bias;
		hi.formatId = format.formatId;
		hi.format = format.format;
		hi.groupId = item.groupId;
		hi.enabled = item.enabled;
		return hi;
	}

	/**
	 * Create HistoryItem description for collecting one variable with one
	 * sampling format. 
	 * 
	 * @param groupItemId
	 *            the group item id to use with
	 *            {@link #composeItemName(String, String, String)}
	 * @param groupId
	 *            the group id to use with
	 *            {@link #composeItemName(String, String, String)}
	 * @param format
	 * @return HistoryItem description.
	 */
	public static SubscriptionItem createItem(
			String groupItemId,
			String groupId,
			SamplingFormat format
			) {
		SubscriptionItem hi = new SubscriptionItem();
		hi.variableId = groupItemId;
		hi.groupItemId = groupItemId;
		hi.id = composeItemName(groupId, groupItemId, format.formatId);
		hi.deadband = format.deadband;
		hi.interval = format.interval;
		hi.formatId = format.formatId;
		hi.format = format.format;
		hi.groupId = groupId;
		hi.enabled = true;
		return hi;
	}

	public static String composeItemName(String groupId, String groupItemId, String formatId)
	{
		return groupId+" "+groupItemId+" "+formatId;
	}

	/** Item identifier in the HistoryManager */
	public @Identifier String id;
		
	/** Variable Id, Reference of variable in Datasource. This field is used by Collector */
	public String variableId;
	
	/** 
	 * Describes the format of the packed sample. The sample must be a record.
	 * The record must have any combination of the following named fields.
	 * The field types must one of: byte, integer, long, float, double.
	 * 
	 * time, endTime, value - are mandatory fields.
	 * 
	 *  time      -  Region start time, the time of the 1st sample included into the band
	 *  endTime   -  Region end time, the time of the last sample included into the band
	 *  
	 *  value     -  First value in the band
	 *  lastValue -  Last value in the band
	 *  avg       -  Average value of all included samples
	 *  median    -  Median value of all samples in the band
	 *  min       -  Lowest value in the band
	 *  max       -  Highest value in the band
	 *  
	 *  quality   -  0 = Good, -1 = No value
	 *  count     -  The number of included samples in the band
	 */
	public Datatype format;
			
	// Subscription parameters
	public double deadband = Double.NaN;
	public double interval = Double.NaN;
	public double gain = 1.0;
	public double bias = 0.0;
	public boolean enabled = true;

	/** Identifier of group or subscription, used by Simantics */
	public String groupId = "";
	
	/** Identifier of the item in the group, used by Simantics */
	public String groupItemId = "";
	
	/** Identifier of the sample format description, used by Simantics */
	public String formatId = "";

}
