package org.simantics.export.core.manager;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;

/**
 * This class exports content.
 *
 * @author toni.kalajainen@semantum.fi
 * @author tuukka.lehtonen@semantum.fi
 */
public class ExportManager {

	public Variant options;
	public ExportContext ctx;

	public ExportManager(Variant options, ExportContext ctx) {
		this.options = options;
		this.ctx = ctx;
	}

	/**
	 * Execute export plan
	 * 
	 * @param ctx
	 *            export context
	 * @param monitor
	 *            the progress monitor to use for reporting progress to the
	 *            user. It is the caller's responsibility to call done() on the
	 *            given monitor. Accepts <code>null</code>, indicating that no
	 *            progress should be reported and that the operation cannot be
	 *            cancelled.</pre>
	 * @param plan
	 *            export plan
	 * @throws ExportException
	 */
	public void execute(ExportContext ctx, IProgressMonitor monitor, ExportPlan plan) throws ExportException
	{
		// 10000 units for all actions and 10 units the cleanup of each
		int totalWork = plan.totalWork(ctx) * 10000 + plan.actions.size() * 100;
		monitor.beginTask( plan.label, totalWork );
		SubMonitor mon = SubMonitor.convert(monitor, plan.label, totalWork);

		try {

			try {
				// 1. Export
				for ( ExportAction action : plan.actions ) {
					if ( monitor.isCanceled() ) return;
					int work = action.work(ctx) * 10000;
					mon.setTaskName( action.label(ctx) );
					action.execute(ctx, mon.newChild(work), options);
				}
			} finally {
				// 2. Cleanup
				for ( ExportAction action : plan.actions ) {
					mon.setTaskName( "Cleanup: " + action.label(ctx) );
					try {
						action.cleanup(ctx, mon.newChild(100), options);
					} catch (ExportException ee) {
						ee.printStackTrace();
					}
				}
			}

		} catch (ExportException ee) {
			monitor.setTaskName( ee.getClass().getName()+": "+ee.getClass().getName() );
			throw ee;
		}
	}

	/**
	 * Validate the plan is executable.
	 * 
	 * @param ctx
	 * @param optionsBinding
	 * @param options
	 * @return null or a label describing the expected problem
	 */
	public List<String> validate(ExportContext ctx, ExportPlan plan)
	{
		List<String> result = new ArrayList<String>(0);
		for ( ExportAction action : plan.actions ) {
			List<String> problems = action.validate(ctx, options);
			result.addAll( problems );
		}
		return result;
	}


}
