/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.flag;

import java.util.Collection;
import java.util.Collections;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.g2d.elementclass.FlagClass;
import org.simantics.g2d.elementclass.FlagClass.Mode;
import org.simantics.g2d.elementclass.FlagClass.Type;
import org.simantics.layer0.utils.binaryPredicates.OrderedSetElementsPredicate;
import org.simantics.structural2.modelingRules.IModelingRules;

/**
 * @author Tuukka Lehtonen
 */
public abstract class AbstractFlagType implements IFlagType {

    protected final Resource flag;
    protected final IModelingRules modelingRules;

    public AbstractFlagType(Resource flag, IModelingRules modelingRules) {
        if (flag == null)
            throw new NullPointerException("null flag");
        if (modelingRules == null)
            throw new NullPointerException("null modeling rules");
        this.flag = flag;
        this.modelingRules = modelingRules;
    }

    protected Type getType(ReadGraph graph) throws DatabaseException {
        DiagramResource dr = DiagramResource.getInstance(graph);
        FlagClass.Type t = DiagramGraphUtil.toFlagType(dr, graph.getPossibleObject(flag, dr.HasFlagType));
        return t;
    }

    protected Mode getMode(ReadGraph graph) throws DatabaseException {
        return getMode(graph, flag);
    }

    public static Mode getMode(ReadGraph graph, Resource flag) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);
        Collection<Resource> connectionJoins = graph.getObjects(flag, DIA.FlagIsJoinedBy);
        int joinCount = connectionJoins.size();
        if (joinCount == 0) {
            return FlagClass.Mode.Internal;
        } else {
            Collection<Resource> flagDiagram = OrderedSetElementsPredicate.INSTANCE.getSubjects(graph, flag);
            for (Resource connectionJoin : connectionJoins)
                for (Resource otherFlag : graph.getObjects(connectionJoin, DIA.JoinsFlag))
                    if (!flag.equals(otherFlag)
                            && Collections.disjoint(
                                    flagDiagram,
                                    OrderedSetElementsPredicate.INSTANCE.getSubjects(graph, otherFlag)))
                        return FlagClass.external(joinCount);
            return FlagClass.internal(joinCount);
        }
    }

}
