/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.debug.ui;

import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.LinkedList;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.handlers.HandlerUtil;
import org.simantics.Simantics;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.browsing.ui.model.browsecontexts.BrowseContext;
import org.simantics.browsing.ui.model.nodetypes.EntityNodeType;
import org.simantics.browsing.ui.model.nodetypes.NodeType;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.ui.selection.WorkbenchSelectionUtils;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class ShowInBrowser extends AbstractHandler {

	private static final String DEFAULT_BROWSER_VIEW_ID = "org.simantics.modeling.ui.browser"; //$NON-NLS-1$

	private static Read<Collection<NodeContext>> getParentsRequest(BrowseContext bc, NodeContext context) {
		return new UniqueRead<Collection<NodeContext>>() {
			@Override
			public Collection<NodeContext> perform(ReadGraph graph) throws DatabaseException {
				return bc.getParents(graph, context);
			}
		};
	}

	private static Read<NodeType> getNodeTypeRequest(Resource element) {
		return new UniqueRead<NodeType>() {
			@Override
			public NodeType perform(ReadGraph graph) throws DatabaseException {
				return EntityNodeType.getNodeTypeFor(graph, element);
			}
		};
	}

	private static Collection<NodeContext> tryGetParents(BrowseContext bc, NodeContext context) {
		try {
			return Simantics.getSession().syncRequest(getParentsRequest(bc, context));
		} catch (DatabaseException e) {
			return Collections.emptyList();
		}
	}

	private static boolean show(GraphExplorer explorer, BrowseContext browseContext, NodeContext context, Deque<NodeContext> path) {
		if (explorer.isVisible(context))
			return explorer.selectPath(path);
		else {
			for (NodeContext parent : tryGetParents(browseContext, context)) {
				path.addFirst(parent);
				if (show(explorer, browseContext, parent, path))
					return true;
				path.removeFirst();
			}
		}
		return false;
	}

	public static Object defaultExecute(ISelection selection, String browserViewId) {
		IViewPart browser = (IViewPart) WorkbenchUtils.findView(browserViewId);
		if (browser == null)
			return null;

		GraphExplorer explorer = (GraphExplorer) browser.getAdapter(GraphExplorer.class);
		BrowseContext browseContext = (BrowseContext) browser.getAdapter(BrowseContext.class);
		if (explorer == null || browseContext == null)
			return null;

		try {
			final Resource element = WorkbenchSelectionUtils.getPossibleResource(selection);
			WorkbenchUtils.showView(browserViewId, IWorkbenchPage.VIEW_VISIBLE);
			NodeType nodeType = Simantics.getSession().syncRequest(getNodeTypeRequest(element));;
			NodeContext context = NodeContextBuilder.buildWithData(NodeType.KEY_SEQUENCE, new Object[] { element, nodeType });
			Deque<NodeContext> path = new LinkedList<>();
			path.add(context);
			if (show(explorer, browseContext, context, path)) {
				WorkbenchUtils.activateView(browserViewId);
			}
		} catch (DatabaseException e) {
		} catch (PartInitException e) {
		}

		return null;
	}

	public static Object defaultExecute(ISelection selection) {
		return defaultExecute(selection, DEFAULT_BROWSER_VIEW_ID);
	}

	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		return defaultExecute(HandlerUtil.getCurrentSelectionChecked(event));
	}

}