/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.procore.cluster;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterI;
import org.simantics.db.impl.IntAllocatorI;
import org.simantics.db.impl.Modifier;

final class TableIntArraySet2 {
    public static final int HeaderSize = 1; 
    private static final int SIZE_OFFSET = -1;
    static int create(final int[] keys, final int[] vals, IntAllocatorI allocator)
    throws DatabaseException {
        final int LENGTH = keys.length;
        if (LENGTH < 1 || LENGTH != vals.length)
            throw new DatabaseException("Illegal argument to create TableIntArraySet2.");
        int newBase = allocator.allocate(LENGTH*2 + HeaderSize) + HeaderSize;
        int[] table = allocator.getTable();
        table[newBase + SIZE_OFFSET] = -LENGTH;
        for (int i=0; i<LENGTH; ++i) {
            if (0 == keys[i])
                throw new DatabaseException("Illegal value to create TableIntArraySet2.");
            table[newBase + i*2] = keys[i];
            table[newBase + i*2 + 1] = vals[i];
        }
        return newBase;
    }
    static boolean isArraySet(int[] table, int base) {
        return table[base + SIZE_OFFSET] < 0;
    }
    static class Tables {
        int[] keys;
        int[] vals;
        public Tables() {
            this.keys = null;
            this.vals = null;
        }
    }
    static Tables getInts(int[] table, int base) {
        final int REAL_SIZE = -table[base + SIZE_OFFSET];
        assert(REAL_SIZE > 0);
        Tables t = new Tables();
        int i;
        for (i=0; i<REAL_SIZE; ++i) {
            int k = table[base + i*2];
            if (0 == k)
                break;
        }
        final int size = i;
        assert(size > 0);
        t.keys = new int[size];
        t.vals = new int[size];
        for (i=0; i<size; ++i){
            t.keys[i] = table[base + i*2];
            t.vals[i] = table[base + i*2 + 1];
        }
        return t;
    }
    static int get(int[] table, int base, int aKey) {
        final int REAL_SIZE = -table[base + SIZE_OFFSET];
        assert(REAL_SIZE > 0);
        for (int i=0; i<REAL_SIZE; ++i, base += 2) {
            if (0 == table[base])
                return 0;
            else if (aKey == table[base]) {
                return table[base + 1];
            }
        }
        return 0;
    }
    /**
     * @param table
     * @param base
     * @param key
     * @param allocator
     * @return new base if object was actually added.
     */
    static int addInt(int[] table, int base, final int key, int val, IntAllocatorI allocator) {
        final int size = -table[base + SIZE_OFFSET];
        assert(size > 0);
        int i;
        for (i=0; i<size; ++i) {
            if (key == table[base + i*2]) {
                if (val == table[base + i*2 + 1])
                    return 0;
                else {
                    table[base + i*2 + 1] = val;
                    return base;
                }
            } else if (0 == table[base + i*2])
                break;
        }
        if (i < size) {
            assert(0 == table[base + i*2]);
            table[base + i*2] = key;
            table[base + i*2 + 1] = val;
            return base;
        }
        final int newSize = size + 1;
        int newBase = allocator.allocate(newSize*2 + HeaderSize) + HeaderSize;
        int[] newTable = allocator.getTable();
        newTable[newBase + SIZE_OFFSET] = -newSize;
        System.arraycopy(table, base, newTable, newBase, size*2);
        newTable[newBase + size*2] = key;
        newTable[newBase + size*2 + 1] = val;
        return newBase;
    }       
    
    static boolean removeInt(int[] table, int base, int key) {
        final int size = -table[base + SIZE_OFFSET];
        assert(size > 0);
        int i;
        for (i=0; i<size; ++i)
            if (key == table[base + i*2])
                break;
            else if (0 == table[base + i*2])
                return false; // not found
        if (i == size)
            return false; // not found
        int end = size - 1;
        for (;end>i; --end)
        	if (0 != table[base + end*2])
        		break;
        table[base + i*2] = table[base + end*2];
        table[base + i*2 + 1] = table[base + end*2 + 1];
        table[base + end*2] = 0;
        table[base + end*2 + 1] = 0;
        return true;
    }

    static int getSize(int[] table, int base) {
        final int size = -table[base + SIZE_OFFSET];
        assert(size > 0);
        int i;
        for (i=0; i<size; ++i)
            if (0 == table[base + i*2])
                break;
        return i;
    }
    
    static int getAllocatedSize(int[] table, int base) {
        final int size = -table[base + SIZE_OFFSET];
        assert(size>0);
        return size + HeaderSize;
    }
    
    static <Context> boolean foreachInt(final int[] table, final int base
    		, final ClusterI.PredicateProcedure<Context> procedure, final Context context, final Modifier modifier) throws DatabaseException {
        final int size = -table[base + SIZE_OFFSET];
        assert(size>0);
        for (int i=0; i<size; ++i) {
            int pRef = table[base + i*2];
            if (0 == pRef)
                break;
            int key;
            if (null == modifier)
                key = pRef;
            else
                key = modifier.execute(pRef);
            if (procedure.execute(context, key, table[base + i*2 + 1]))
                return true; // loop broken by procedure
        }
        return false; // loop finished
    }
}

