package org.simantics.db.impl.query;

import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.simantics.db.ObjectResourceIdMap;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterBase;
import org.simantics.db.impl.ClusterSupport;
import org.simantics.db.impl.ClusterTraitsBase;
import org.simantics.db.service.Bytes;
import org.slf4j.LoggerFactory;

import gnu.trove.map.hash.TIntLongHashMap;

public class QueryDeserializer {

    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(QueryDeserializer.class);

    QueryCache qc;
    QuerySupport qs;
    ClusterSupport cs;
    
    private byte[] bytes;
    private int byteIndex;
    
    private TIntLongHashMap clusterKeys = new TIntLongHashMap();
    private Map<Integer,QueryFactory> ids = new HashMap<Integer,QueryFactory>();

    public QueryDeserializer(QueryProcessor qp, byte[] bytes) {
        this.qc = qp.cache;
        this.qs = qp.querySupport;
        this.cs = qs.getClusterSupport();;
        this.bytes = bytes;
    }
    
    public byte readByte() {
        return bytes[byteIndex++];
    }
    
    public int readLE4() {
        int result = Bytes.readLE4(bytes, byteIndex);
        byteIndex += 4;
        return result;
    }
    
    public long readLE8() {
        long result = Bytes.readLE8(bytes, byteIndex);
        byteIndex += 8;
        return result;
    }

    public byte[] readBytes(int len) {
        byte[] result = Arrays.copyOfRange(bytes, byteIndex, byteIndex+len);
        byteIndex += len;
        return result;
    }
    
    public void readHeaders() {
        int idsSize = readLE4();
        for(int i=0;i<idsSize;i++) {
            int size = readLE4();
            byte[] data = readBytes(size);
            String id = new String(data);
            int key = readLE4();
            try {
                Class<QueryFactory> clazz = (Class<QueryFactory>)getClass().getClassLoader().loadClass(id + "Factory");
                QueryFactory qf = clazz.getDeclaredConstructor().newInstance(); 
                ids.put(key, qf);
            } catch (ClassNotFoundException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (InstantiationException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (IllegalAccessException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (IllegalArgumentException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (InvocationTargetException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (NoSuchMethodException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (SecurityException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            }
        }
        int clusterKeysSize = readLE4();
        for(int i=0;i<clusterKeysSize;i++) {
            long cluster = readLE8();
            int key = readLE4();
            clusterKeys.put(key, cluster);
        }
    }
    
    public QueryFactory readFactory() {
        int key = readLE4();
        return ids.get(key);
    }

    public void readQueries() {
        int count = readLE4();
        for(int i=0;i<count;i++) {
            QueryFactory qf = readFactory();
            try {
                qf.read(this);
            } catch (DatabaseException e) {
                e.printStackTrace();
            }
        }
    }

    public int readResource() throws DatabaseException {
        int key = readLE4();
        if(key < 0)
            return key;
        int clusterKey = ClusterTraitsBase.getClusterKeyFromResourceKey(key);
        long cluster = clusterKeys.get(clusterKey);
        ClusterBase cb = cs.getClusterByClusterId(cluster);
        return ClusterTraitsBase.createResourceKey(cb.getClusterKey(), ClusterTraitsBase.getResourceIndexFromResourceKey(key));
    }
    
    public byte[] readByteArray() {
        int len = readLE4();
        if(len == -1)
            return null;
        return readBytes(len);
    }
    
    public String readString() {
        return new String(readByteArray());
    }
    
    public ObjectResourceIdMap<String> createChildMap() {
        return qs.createChildMap();
    }

    AssertedPredicates readAssertedPredicates() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateAssertedPredicates(r);
    }

    AssertedStatements readAssertedStatements() throws DatabaseException {
        int r1 = readResource();
        int r2 = readResource();
        return qc.getOrCreateAssertedStatements(r1, r2);
    }

    ChildMap readChildMap() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateChildMap(r);
    }

    DirectObjects readDirectObjects() throws DatabaseException {
        int r1 = readResource();
        int r2 = readResource();
        return qc.getOrCreateDirectObjects(r1, r2);
    }
    
    DirectPredicates readDirectPredicates() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateDirectPredicates(r);
    }

    Objects readObjects() throws DatabaseException {
        int r1 = readResource();
        int r2 = readResource();
        return qc.getOrCreateObjects(r1, r2);
    }

    OrderedSet readOrderedSet() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateOrderedSet(r);
    }

    Predicates readPredicates() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreatePredicates(r);
    }

    PrincipalTypes readPrincipalTypes() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreatePrincipalTypes(r);
    }

    RelationInfoQuery readRelationInfoQuery() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateRelationInfoQuery(r);
    }

    Statements readStatements() throws DatabaseException {
        int r1 = readResource();
        int r2 = readResource();
        return qc.getOrCreateStatements(r1, r2);
    }

    SuperRelations readSuperRelations() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateSuperRelations(r);
    }

    SuperTypes readSuperTypes() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateSuperTypes(r);
    }

    TypeHierarchy readTypeHierarchy() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateTypeHierarchy(r);
    }

    Types readTypes() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateTypes(r);
    }

    URIToResource readURIToResource() throws DatabaseException {
        String s = readString();
        return qc.getOrCreateURIToResource(s);
    }

    ValueQuery readValueQuery() throws DatabaseException {
        int r = readResource();
        return qc.getOrCreateValueQuery(r);
    }

}
