/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.Arrays;

import org.simantics.db.exception.DatabaseException;


final public class IntArray {

	public static IntArray EMPTY = new IntArray();
	
    public int[] data;

    /** the index after the last entry in the list */
    public int sizeOrData;

    /** the default capacity for new lists */
    protected static final int DEFAULT_CAPACITY = 3;

    protected static final int NO_DATA = -1;

    public IntArray() {
        data = null;
        sizeOrData = NO_DATA;
    }

    /**
     * Returns the number of values in the list.
     *
     * @return the number of values in the list.
     */
    public int size() {
        return data != null ? sizeOrData : (sizeOrData != NO_DATA ? 1 : 0);
    }

    /**
     * Tests whether this list contains any values.
     *
     * @return true if the list is empty.
     */
    public boolean isEmpty() {
        return sizeOrData == NO_DATA;
    }

//    /**
//     * Sheds any excess capacity above and beyond the current size of
//     * the list.
//     */
//    public void trimToSize() {
//        if (_data.length > size()) {
//            int[] tmp = new int[size()];
//            toNativeArray(tmp, 0, tmp.length);
//            _data = tmp;
//        }
//    }

    // modifying

    public void trim() {
    	if(data != null) {
    		int[] newData = new int[sizeOrData];
    		System.arraycopy(data, 0, newData, 0, sizeOrData);
    		data = newData;
    	}
    }
    
    /**
     * Adds <tt>val</tt> to the end of the list, growing as needed.
     *
     * @param val an <code>int</code> value
     */
    public void add(int val) {
        if(data == null) {
            if(sizeOrData == NO_DATA) {
                sizeOrData = val;
            } else {
                data = new int[DEFAULT_CAPACITY];
                data[0] = sizeOrData;
                data[1] = val;
                sizeOrData = 2;
            }
        } else {
            if(data.length == sizeOrData) {
                int newCap = data.length << 1;
                int[] tmp = new int[newCap];
                System.arraycopy(data, 0, tmp, 0, data.length);
                data = tmp;
                data[sizeOrData++] = val;
            } else {
                data[sizeOrData++] = val;
            }
        }
    }
    
    int[] toArray() {
    	int size = size();
    	int[] result = new int[size];
    	if(size == 1) {
    		result[0] = sizeOrData;	
    	} else {
            System.arraycopy(data, 0, result, 0, size);
    	}
    	return result;
    }
 
    public void serialize(QuerySerializer serializer) {
        int size = size();
        serializer.writeLE(size);
        if(size == 1) {
            serializer.addResource(sizeOrData);
        } else {
            for(int i=0;i<size;i++)
                serializer.addResource(data[i]);
        }
    }
 
    public static IntArray deserialize(QueryDeserializer deserializer) throws DatabaseException {
        int size = deserializer.readLE4();
        IntArray result = new IntArray();
        for(int i=0;i<size;i++) {
            result.add(deserializer.readResource());
        }
        return result;
    }

    @Override
    public boolean equals(Object object) {
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (IntArray.class != object.getClass())
            return false;
        IntArray r = (IntArray)object;
//        System.out.println("equals " + this + " vs. " + r);
        return sizeOrData == r.sizeOrData && Arrays.equals(data, r.data);
    }
    
    @Override
    public String toString() {
        return "IntArray " + sizeOrData + " " + Arrays.toString(data);
    }
    
}