/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import junit.framework.TestCase;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.annotations.Length;
import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.annotations.Pattern;
import org.simantics.databoard.annotations.Range;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.error.BindingException;

public class TestValidator extends TestCase {

	public enum Test { Key1, Key2 }
	
	public static class Luokka {
		public @Range("[0..100]") Integer percent = 0;
		public @Optional Integer optionalValue;
		public Boolean booleanValue = true;
		public @Length("7") @Range("[1..39]") int[] lottoNumbers = new int[] {1,2,3,4,5,6,7};
		public @Length("10") @Pattern("0x\\d{8}") String hexNumber = "0x11223344"; 
	}
	
	public static Binding b = Bindings.getBindingUnchecked(Luokka.class);
	public Luokka l;
	
	public void setUp() throws Exception {
		l = new Luokka();
	}
	
	public void testNumberType() throws Exception {
		
		l.percent = 100; 
		b.assertInstaceIsValid( l );
		
		l.percent = 0;
		b.assertInstaceIsValid( l );
		
		try {
			l.percent = -1;
			b.assertInstaceIsValid( -1 );
			fail();
		} catch ( BindingException e ) {}
		
		try {
			l.percent = 101;
			b.assertInstaceIsValid( 101 );
			fail();
		} catch ( BindingException e ) {}

		try {
			b.assertInstaceIsValid( new Integer(5) );
			fail();
		} catch ( BindingException e ) {}		
		
	}
	
	public void testOptionalType() throws BindingException {
		l.optionalValue = 5;
		b.assertInstaceIsValid( l );
		
		try {
			l.percent = null;
			b.assertInstaceIsValid( l );
			fail();
		} catch ( BindingException e ) {}				
	}
	
	public void testBooleanType() throws BindingException {
		b.assertInstaceIsValid( l );
		
		try {
			b.assertInstaceIsValid( new Integer(5) );
			fail();
		} catch ( BindingException e ) {}				
		
	}
	
	public void testStringType() throws BindingException {
		b.assertInstaceIsValid( l );
		
		try {
			// not according to pattern
			l.hexNumber = "Hello World";
			b.assertInstaceIsValid( l );
			fail();
		} catch ( BindingException e ) {}
		
		try {
			// Too short
			l.hexNumber = "0x100";
			b.assertInstaceIsValid( l );
			fail();
		} catch ( BindingException e ) {}				
		
	}
	
	public void testRecordType() throws BindingException {
		b.assertInstaceIsValid( l );
		// Already thoroughly tested by all other tests
	}
	
	public void testUnionType() throws BindingException, BindingConstructionException {
		Binding eb = Bindings.getBinding( Test.class );
		eb.assertInstaceIsValid(Test.Key1);
		eb.assertInstaceIsValid(Test.Key2);
		try {
			b.assertInstaceIsValid( new Integer(5) );
			fail();
		} catch ( BindingException e ) {}				
		
	}
	
	public void testArrayType() throws BindingException {
		b.assertInstaceIsValid( l );
				
		try {
			// Too many numbers
			l.lottoNumbers = new int[] {1,2,3,4,5,6,7, 10, 10, 10, 10};
			b.assertInstaceIsValid( l );
			fail();
		} catch ( BindingException e ) {}				

		try {
			// Not enough numbers
			l.lottoNumbers = new int[] {1, 2, 3};
			b.assertInstaceIsValid( l );
			fail();
		} catch ( BindingException e ) {}				
				
		try {
			// not with in 1..39
			l.lottoNumbers = new int[] {100,-2,3,4,5,6,7};
			b.assertInstaceIsValid( l );
			fail();
		} catch ( BindingException e ) {}

	}
	
}

