/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.Random;

import org.junit.Test;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.MapBinding;
import org.simantics.databoard.binding.OptionalBinding;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.UnionBinding;
import org.simantics.databoard.binding.VariantBinding;
import org.simantics.databoard.binding.util.RandomValue;
import org.simantics.databoard.type.Datatype;

/**
 * Test Clone is a test case that asserts that {@link Bindings#clone(Object, Binding, Binding)}
 * returns two separate instances.
 * 
 * Also tests that adapt of same binding returns the same instance. 
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class TestClone {
	
	public @Test void testGenericBinding() throws Exception {		
		RandomValue rv = new RandomValue();
		rv.refereableRecords = true;
		for (int i=0; i<10000; i++) {
			rv.random = new Random(i);
			rv.random.nextLong();
			Datatype type = rv.randomType(0, 3);
			Binding binding = Bindings.getMutableBinding(type);
			Object o1 = binding.accept(rv);
			Object o2 = binding.clone(o1);
			System.out.println(i+": "+type);
			if (binding.equals(o1, o2)) continue;
			assertNotSameInstance(binding, o1, o2);
			
			Object o3 = Bindings.adapt(o1, binding, binding);
			assertTrue( o3 == o1 );
		}		
	}
	
	public void assertNotSameInstance(Binding b, Object o1, Object o2) throws Exception {
		assertTrue(b.isImmutable() || o1 != o2);
		
		if (b instanceof RecordBinding) assertNotSameRecord((RecordBinding) b, o1, o2);
		if (b instanceof UnionBinding) assertNotSameUnion((UnionBinding) b, o1, o2);
		if (b instanceof ArrayBinding) assertNotSameArray((ArrayBinding) b, o1, o2);
		if (b instanceof MapBinding) assertNotSameMap((MapBinding) b, o1, o2);
		if (b instanceof OptionalBinding) assertNotSameOptional((OptionalBinding) b, o1, o2);
		if (b instanceof VariantBinding) assertNotSameVariant((VariantBinding) b, o1, o2);
	}
	
	public void assertNotSameArray(ArrayBinding b, Object o1, Object o2) throws Exception {
		int count = b.size(o1);
		Binding cb = b.getComponentBinding();
		for (int i=0; i<count; i++) {
			Object v1 = b.get(o1, i);
			Object v2 = b.get(o2, i);
			assertNotSameInstance(cb, v1, v2);
		}
	}

	public void assertNotSameMap(MapBinding b, Object o1, Object o2) throws Exception {
		int count = b.size(o1);
		Binding vb = b.getValueBinding();
		Object values1[] = b.getValues(o1);  
		Object values2[] = b.getValues(o2);  
		for (int i=0; i<count; i++) {
			Object v1 = values1[i];
			Object v2 = values2[i];
			assertNotSameInstance(vb, v1, v2);
		}		
	}

	public void assertNotSameOptional(OptionalBinding b, Object o1, Object o2) throws Exception {
		Binding cb = b.getComponentBinding();
		Object v1 = b.getValue(o1);
		Object v2 = b.getValue(o2);
		assertNotSameInstance(cb, v1, v2);
	}

	public void assertNotSameVariant(VariantBinding b, Object o1, Object o2) throws Exception {
		Binding cb = b.getContentBinding(o1);
		Object v1 = b.getContent(o1);
		Object v2 = b.getContent(o2);
		assertNotSameInstance(cb, v1, v2);		
	}

	public void assertNotSameRecord(RecordBinding b, Object o1, Object o2) throws Exception {
		int count = b.getComponentCount();
		for (int i=0; i<count; i++) {
			Binding cb = b.getComponentBinding(i);
			Object v1 = b.getComponent(o1, i);
			Object v2 = b.getComponent(o2, i);
			assertNotSameInstance(cb, v1, v2);
		}	
	}

	public void assertNotSameUnion(UnionBinding b, Object o1, Object o2) throws Exception {
		int tag = b.getTag(o1);
		assertEquals(tag, b.getTag(o2));
		Binding cb = b.getComponentBinding(tag);
		Object v1 = b.getValue(o1);
		Object v2 = b.getValue(o2);		
		assertNotSameInstance(cb, v1, v2);
	}

}

