/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.util;

import org.simantics.databoard.annotations.Union;

@Union({
	Limit.Nolimit.class,
	Limit.Inclusive.class, Limit.Exclusive.class,
	Limit.InclusiveLong.class, Limit.ExclusiveLong.class
})
public abstract class Limit {

	private static Limit NOLIMIT = new Nolimit();

	public static Limit nolimit() { return NOLIMIT; }
	public static Limit inclusive(Byte value) { return value==null ? NOLIMIT : new InclusiveLong(value.longValue()); }
	public static Limit inclusive(Integer value) { return value==null ? NOLIMIT : new InclusiveLong(value.longValue()); }
	public static Limit inclusive(Long value) { return value==null ? NOLIMIT : new InclusiveLong(value); }
	public static Limit inclusive(Float value) { return value==null ? NOLIMIT : new Inclusive(value.doubleValue()); }
	public static Limit inclusive(Double value) { return value==null ? NOLIMIT : new Inclusive(value); }
	public static Limit exclusive(Byte value) { return value==null ? NOLIMIT : new ExclusiveLong(value.longValue()); }
	public static Limit exclusive(Integer value) { return value==null ? NOLIMIT : new ExclusiveLong(value.longValue()); }	
	public static Limit exclusive(Long value) { return value==null ? NOLIMIT : new ExclusiveLong(value); }
	public static Limit exclusive(Float value) { return value==null ? NOLIMIT : new Exclusive(value.doubleValue()); }	
	public static Limit exclusive(Double value) { return value==null ? NOLIMIT : new Exclusive(value); }	
	
	public abstract boolean isInclusive();
	public abstract boolean isExclusive();
	public abstract Number getValue();
	public abstract Integer smallestIncludedInteger();
	public abstract Integer greatestIncludedInteger();
	Limit() {}
	
	public static class Nolimit extends Limit {
		public Nolimit() {}
		public Number getValue() {return null;}
		public boolean isExclusive() {return false;}
		public boolean isInclusive() {return false;}
		public Integer greatestIncludedInteger() { return null;	}
		public Integer smallestIncludedInteger() { return null;	}
	}
	
	public static class Inclusive extends Limit {
		public Double value;		
		public Inclusive(Double value) { this.value = value; }
		public Number getValue() { return value; }
		public boolean isExclusive() { return false; }
		public boolean isInclusive() { return true; }
		public int hashCode() { return value.hashCode() ^ -1; }
		public Integer greatestIncludedInteger() { 
			return (int)Math.floor(value.doubleValue());	
		}
		public Integer smallestIncludedInteger() { 
			return (int)Math.ceil(value.doubleValue());	
		}
	}

	public static class Exclusive extends Limit {
		public Double value;		
		public Exclusive(Double value) { this.value = value; }
		public Number getValue() { return value; }
		public boolean isExclusive() { return true; }
		public boolean isInclusive() { return false; }		
		public int hashCode() { return value.hashCode(); }
		public Integer greatestIncludedInteger() { 
			return -1+(int)Math.ceil(value.doubleValue());	
		}
		public Integer smallestIncludedInteger() { 
			return 1+(int)Math.floor(value.doubleValue());	
		}
	}
	public static class InclusiveLong extends Limit {
		public Long value;		
		public InclusiveLong(Long value) { this.value = value; }
		public Number getValue() { return value; }
		public boolean isExclusive() { return false; }
		public boolean isInclusive() { return true; }
		public int hashCode() { return new Double(value.doubleValue()).hashCode() ^ -1; }
		public Integer greatestIncludedInteger() { 
			return value.intValue();	
		}
		public Integer smallestIncludedInteger() { 
			return value.intValue();	
		}
	}

	public static class ExclusiveLong extends Limit {
		public Long value;		
		public ExclusiveLong(Long value) { this.value = value; }
		public Number getValue() { return value; }
		public boolean isExclusive() { return true; }
		public boolean isInclusive() { return false; }
		public int hashCode() { return new Double(value.doubleValue()).hashCode(); }
		public Integer greatestIncludedInteger() { 
			return value.intValue()-1;	
		}
		public Integer smallestIncludedInteger() { 
			return value.intValue()+1;	
		}
	}

	/**
	 * Compares two limits for equality.
	 * 
	 * Integer and flot are equal, e.g. [5 is equal to [5.0
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj instanceof Limit == false) return false;
		Limit other = (Limit) obj;
		Number ln = getValue();
		Number on = other.getValue();		
		return isInclusive()==other.isInclusive() && isExclusive()==other.isExclusive() &&
			( ln==null ? on==null : on==null ? false : NumberComparator.INSTANCE.compare(ln, on)==0 );
	}
	
	@Override
	public String toString() {
		return getValue().toString();
	}	
	
}

