/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.type;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import org.simantics.databoard.accessor.error.ReferenceException;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.accessor.reference.IndexReference;
import org.simantics.databoard.accessor.reference.LabelReference;
import org.simantics.databoard.accessor.reference.NameReference;
import org.simantics.databoard.annotations.Referable;
import org.simantics.databoard.binding.error.DatatypeConstructionException;
import org.simantics.databoard.util.IdentityPair;

public @Referable class RecordType extends Datatype {
	
	public static final Datatype VOID_TYPE = new RecordType(false);
	
	public static final String KEY_REFERABLE = "referable"; // "false"/"true"
	
	// Key to identifier(s) index
	public static final String KEY_IDENTIFIER = "identifier"; // "0" "0,1" "n[,m]"
	
	public static final Component[] NO_COMPONENTS = new Component[0];
	
    Component[] components = NO_COMPONENTS;
    
    /** Indices to identifiers of this record. This field is filled on request */
    private transient int[] identifiersIndices;
    
    /** Identifier type */
    private transient Datatype identifierType;
    
    public RecordType() {    	
    	setReferable(false);
    }
    
    public RecordType(boolean referable, Component...components) {
    	this.components = components;
    	setReferable(referable);
    }
    
    public boolean isReferable() {
    	String str = metadata.get( KEY_REFERABLE );
    	return str!=null && str.equals( Boolean.TRUE.toString() );
    }
    
    public void setReferable( boolean referable ) {
    	if ( !referable ) 
    	{
    		metadata.remove(KEY_REFERABLE);  
    	} else {
    		metadata.put(KEY_REFERABLE, Boolean.toString(referable));
    	}
    }

    @Override
    protected void collectSubtypes(Set<Datatype> subtypes,
    		Set<Datatype> recursiveSubtypes) {
    	if(!subtypes.add(this)) {
    		recursiveSubtypes.add(this);
    		return;
    	}
    	for(Component component : components)
	        component.type.collectSubtypes(subtypes, recursiveSubtypes);
    }
    
    public void setComponents(Component[] components) {
    	this.components = components;
    }
   
    public void mergeRecord(RecordType src)
    throws DatatypeConstructionException
    {
    	int ci = src.getComponentCount();
    	for (int i=0; i<ci; i++) {
    		Component sc = src.components[ i ];
    		
    		int li = getComponentIndex2( sc.name );
    		if ( li<0 ) {
    			addComponent(sc.name, sc.type);
    		} else {
				Component lc = components[ li ];
    			if ( sc.type instanceof RecordType && lc.type instanceof RecordType ) {
    				((RecordType)lc.type).mergeRecord( (RecordType) sc.type );
    			} else if ( sc.type.equals( lc.type ) ) {} 
    			else {
    				throw new DatatypeConstructionException("Cannot merge field \""+sc.name+"\" "+sc.type.getClass().getName()+" and "+lc.getClass().getName());
    			}
    		}
    		
    	}
    }
    
    public void addComponent(String name, Datatype type)
    {
    	Component c = new Component(name, type);
    	if (components == null) {
    		components = new Component[] { c };
    	} else {
    		Component[] newComponents = new Component[ components.length +1 ];
    		System.arraycopy(components, 0, newComponents, 0, components.length);
    		newComponents[ components.length ] = c;
    		components = newComponents;    	
    	}    	
    }
    
    public void clear() {
    	components = new Component[0];
    	metadata.clear();
    }
    
    public void removeComponent(String name) {
    	int index = getComponentIndex2(name);
    	if (index<0) throw new IllegalArgumentException();
		Component[] newComponents = new Component[ components.length -1 ];
		if (index>0) System.arraycopy(components, 0, newComponents, 0, index);
		if (index<newComponents.length) System.arraycopy(components, index+1, newComponents, index, newComponents.length - index);
		components = newComponents;    	
		// xxx untested
    }
    
    @Override
    public int getComponentCount() {
    	return components.length;
    }
    
    @Override
    protected boolean deepEquals(Object obj, Set<IdentityPair<Datatype, Datatype>> compareHistory) {
		if ( this==obj ) return true;
		if ( !hasEqualMetadata(obj) ) return false;
		if (obj instanceof RecordType == false) return false;
		RecordType other = (RecordType) obj;
		
		if (components.length!= other.components.length) return false;
		// Verify names
		for (int i = 0; i<components.length; i++) {
			Component lc = components[i];
			Component rc = other.components[i];
			if (!lc.name.equals(rc.name)) return false;
			
		}

		// Verify types
		if (compareHistory==null) compareHistory = new HashSet<IdentityPair<Datatype, Datatype>>(1);

		IdentityPair<Datatype, Datatype> pair = new IdentityPair<Datatype, Datatype>(this, other);
		if (compareHistory.contains(pair)) return true;
		compareHistory.add(pair);
		
		for (int i = 0; i<components.length; i++) {
			Component lc = components[i];
			Component rc = other.components[i];
			if (!lc.type.deepEquals(rc.type, compareHistory)) return false;
		}
		return true;
	}
	
	@Override
	public int hashCode() {
		int hash = super.hashCode();		
		for (Component c : components) 
			hash = hash*13 + 7 * c.name.hashCode() /*+ 3*c.type.hashCode()*/;
		return hash;
	}    
	
	@Override
	public void accept(Visitor1 v, Object obj) {
	    v.visit(this, obj);        
	}

	@Override
	public <T> T accept(Visitor<T> v) {
	    return v.visit(this);
	}
	
	/**
	 * Return true if the record is a tuple.
	 * Tuple is a record with all components are named as a number, the index number of the field.
	 * Empty record is a tuple  
	 * 
	 * @return true if the record type is a tuple.
	 */
	public boolean isTupleType() {
		if (components.length==0) return false;
		for (int i=0; i<getComponentCount(); i++) {
			if (!getComponent(i).name.equals(Integer.toString(i))) return false;
		}
		return true;
	}

    /**
     * Get component type by index
     * @param index index
     * @return componenet type or <tt>null</tt> if index was invalid
     */
    @Override
    public Datatype getComponentType(int index) {
    	if (index<0||index>=components.length) return null;
    	return components[index].type;
    }
    
	@Override
	public Datatype getComponentType(ChildReference path) throws IllegalArgumentException {
		if (path==null) return this;
		if (path instanceof IndexReference) {
			IndexReference ir = (IndexReference) path;
			return components[ir.index].type.getComponentType(path.childReference);
		}
		if (path instanceof NameReference) {
			NameReference nr = (NameReference) path;
			return getComponent( nr.name ).type.getComponentType(path.childReference);
		}
		if (path instanceof LabelReference) {
			LabelReference lr = (LabelReference) path;			
			try {
				Integer i = new Integer(lr.label);
				return getComponent( i ).type.getComponentType(path.childReference);
			} catch (NumberFormatException nfe) {
				return getComponent( lr.label ).type.getComponentType(path.childReference);
			}
		}
		throw new IllegalArgumentException();
	}
    
	public boolean hasComponent(String fieldName) {
        for (int i=0; i<components.length; i++)
            if (components[i].name.equals(fieldName)) return true;
        return false;
	}
	
    /**
     * Get component by name.
     * 
     * @param fieldName component name
     * @return component index or <code>null</code> if one does not exist
     */
    public Integer getComponentIndex(String fieldName) {
        for (int i=0; i<components.length; i++)
            if (components[i].name.equals(fieldName)) return i;
        return null;
    }

    /**
     * Get component by name.
     * 
     * @param fieldName component name
     * @return component index or -1 if one does not exist
     */
    public int getComponentIndex2(String fieldName) {
        for (int i=0; i<components.length; i++)
            if (components[i].name.equals(fieldName)) return i;
        return -1;
    }
    
    /**
     * Get component Datatype by field name
     * @param fieldName
     * @return datatype or <code>null</code>
     */
    public Datatype getComponentType(String fieldName) {
    	int index = getComponentIndex2(fieldName);
    	if (index<0) return null;
    	return components[index].type;
    }
    
    /**
     * Get component by name.
     * 
     * @param fieldName component name
     * @return component or <code>null</code> if one does not exist
     */
    public Component getComponent(String fieldName) {
        for (Component c : components)
            if (c.name.equals(fieldName)) return c;
        return null;
    }

    /**
     * Get component by index.
     * 
     * @param index component index
     * @return component or <code>null</code> if one does not exist
     */
    public Component getComponent(int index) {
    	if (index<0||index>=components.length) return null;
    	return components[index];
    }
    	
	public Component[] getComponents() {
		return components;
	}

	/**
	 * Get an array of indices that describe which fields compose the identifier 
	 * of this record 
	 * 
	 * @return indices
	 */
	public int[] getIdentifiers() {
		String ids = metadata.get( KEY_IDENTIFIER );
		if (ids == null) {
			identifiersIndices = new int[0];
		} else {
			// Parse
			StringTokenizer st = new StringTokenizer(ids, ",");
			
			int[] indices = new int[ st.countTokens() ];
			for (int i=0; i<indices.length; i++) {
				String token = st.nextToken();
				try {
					indices[i] = Integer.valueOf(token);
				} catch ( NumberFormatException nfe ) {
					indices[i] = -1;
				}
			}
			identifiersIndices = indices;
		}
		return identifiersIndices;
	}
	
	/**
	 * Set which fields compose the identifier of this record 
	 * 
	 * @param indices
	 */
	public void setIdentifiers(int...indices)
	{
		if (indices.length==0) {
			metadata.remove( KEY_IDENTIFIER );
			return;
		}
		identifiersIndices = indices;
		StringBuilder sb = new StringBuilder();
		for (int i=0; i<indices.length; i++) {
			if (i>0) sb.append(',');
			sb.append( Integer.toString(indices[i]) );
		}
		
		String str = sb.toString();
		if ( str.isEmpty() ) {
			metadata.remove( KEY_IDENTIFIER ); 
		} else {
			metadata.put( KEY_IDENTIFIER, str );
		}
	}

	/**
	 * Set which fields compose the identifier of this record
	 * @param indices
	 */
	public void setIdentifiers(List<Integer> indices)
	{
		int[] indices2 = new int[indices.size()];
		for (int i=0; i<indices.size(); i++) indices2[i] = indices.get(i);
		setIdentifiers(indices2);
	}
	
	public boolean isIdentifier( int fieldIndex )
	{
		int[] ids = getIdentifiers();
		if (ids == null) return false;
		for (int index : ids)
		{
			if (index == fieldIndex) return true;
		}
		return false;
	}
	
	/**
	 * Get a datatype that describes the identifier of this type.
	 * If no field has Identifier annotation, the result is null.
	 * If more than one field is an identifier the type is a record
	 * with all composing fields.
	 * 
	 * @return identifier type or null
	 */
	public Datatype getIdentifierType() 
	{		
		if ( identifierType != null ) return identifierType;
		
		int[] ids = getIdentifiers();
		if (ids.length==0) return null;
		
		if (ids.length==1) {
			identifierType = getComponentType(ids[0]);
		}
		
		RecordType rt = new RecordType();
		for (int i : ids) {
			Component c = getComponent(i);
			rt.addComponent( c.name, c.type );
		}
		identifierType = rt;		
		return identifierType;
	}

	@SuppressWarnings("unchecked")
	public <T extends Datatype> T getChildType( ChildReference reference ) throws ReferenceException
	{
		if (reference==null) return (T) this;
		
		if (reference instanceof LabelReference) {
			LabelReference lr = (LabelReference) reference;
			String fieldName = lr.label;
			int index = getComponentIndex2(fieldName);
			if (index<0) throw new ReferenceException("RecordType doesn't have field by name \""+fieldName+"\"");
			return components[index].type.getChildType(reference.childReference);
		}		
		
		if (reference instanceof IndexReference) {
			IndexReference ref = (IndexReference) reference;
			int index = ref.getIndex();
			if ( index<0 || index>=components.length ) new ReferenceException("RecordType doesn't have field at index+"+index);
			return components[index].type.getChildType(reference.childReference);
		} 
		
		if (reference instanceof NameReference) {
			NameReference lr = (NameReference) reference;
			String fieldName = lr.name;
			int index = getComponentIndex2(fieldName);
			if (index<0) throw new ReferenceException("RecordType doesn't have field by name \""+fieldName+"\"");
			return components[index].type.getChildType(reference.childReference);
		} 
		
		throw new ReferenceException(reference.getClass()+" is not a subreference of RecordType");
		
	}
}
