/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.type;

import org.simantics.databoard.Units;
import org.simantics.databoard.accessor.error.ReferenceException;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.util.Range;
import org.simantics.databoard.util.RangeException;

/**
 * NumberType is comparable primitive type:
 *  ByteType, IntegerType, DoubleType, LongType or FloatType 
 * 
 */
public abstract class NumberType extends Datatype {
    	
	/** Unit string describes the quantity, magnitude and unit of the value. */
	public static final String KEY_UNIT = "unit";
	
	/** Value ranges that are valid. */
	public static final String KEY_RANGE = "range";
	
	// Cached Range
	private transient Range range;
	private transient String rangeIsForStr;
		
	/**
	 * Get the unit type.
	 * 
	 * Unit string describes the quentity, magnity and unit of the value.
	 *
	 * @return the unit
	 * @see Units 
	 */
	public String getUnit() {
		return metadata.get( KEY_UNIT );
	}

	/**
	 * Set the unit type.
	 * Unit string describes the quentity, magnity and unit of the value.
	 * 	
	 * @param unit the unit type
	 * @see Units 
	 */
	public void setUnit(String unit) {
		if (unit==null) metadata.remove( KEY_UNIT ); 
		else metadata.put( KEY_UNIT, unit );
	}
  
	public Range getRange() {
		String rangeStr = metadata.get( KEY_RANGE );
		if (rangeStr == null) return null;
		if (range != null && rangeStr!=null && rangeStr==rangeIsForStr) return range;
		try {
			rangeIsForStr = rangeStr;
			range = Range.valueOf( rangeStr );
		} catch (RangeException e) {
			range = null;
		}
		return range;
	}
	
	public String getRangeStr() {
		return metadata.get( KEY_RANGE );
	}

	public void setRange(String range) {
		if (range==null) metadata.remove( KEY_RANGE ); else
    	metadata.put( KEY_RANGE, range );
	}
	
	public void setRange(Range range) {
		this.range = range;
		if (range==null) {
			rangeIsForStr = null;
			metadata.remove( KEY_RANGE );
		} else {
			rangeIsForStr = range.toString();
			metadata.put( KEY_RANGE, rangeIsForStr );
		}
	}
	
    @Override
    public int getComponentCount() {
    	return 0;
    }
    
    @Override
    public Datatype getComponentType(int index) {
    	throw new IllegalArgumentException();
    }
    
    @Override
    public Datatype getComponentType(ChildReference path) {
    	if (path==null) return this;
    	throw new IllegalArgumentException();
    }	

	@SuppressWarnings("unchecked")
	@Override
	public <T extends Datatype> T getChildType(ChildReference reference) throws ReferenceException {
		if (reference==null) return (T) this;
		throw new ReferenceException(reference.getClass()+" is not a subreference of NumberType");	
	}
    
}

