/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.type;

import java.util.Set;

import org.simantics.databoard.accessor.error.ReferenceException;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.accessor.reference.IndexReference;
import org.simantics.databoard.accessor.reference.KeyReference;
import org.simantics.databoard.accessor.reference.LabelReference;
import org.simantics.databoard.accessor.reference.NameReference;
import org.simantics.databoard.util.IdentityPair;
import org.simantics.databoard.util.ObjectUtils;

public class MapType extends Datatype {

	/** Key to describe if map is to be serialized ordered, the value is boolean value "false"/"true" */
	public static final String KEY_ORDERED = "ordered";
	
	public Datatype keyType;
	public Datatype valueType;
	
	public MapType() {		
	}
	
	public MapType(Datatype keyType, Datatype valueType) {
		if (keyType==null || valueType==null)
			throw new IllegalArgumentException("Argument must not be null");
		this.keyType = keyType;
		this.valueType = valueType;
	}
	
    @Override
    public int getComponentCount() {
    	return 2;
    }
    
    @Override
    public Datatype getComponentType(int index) {
    	if (index==0) return keyType;
    	if (index==1) return valueType;
    	throw new IllegalArgumentException();
    }
    
    @Override
    public Datatype getComponentType(ChildReference path) {
    	if (path==null) return this;
    	if (path instanceof IndexReference) {
    		IndexReference ir = (IndexReference) path;
    		if (ir.index==0) return keyType.getComponentType(path.childReference);
    		if (ir.index==1) return valueType.getComponentType(path.childReference);
    	}
    	if (path instanceof LabelReference) {
    		LabelReference lr = (LabelReference) path;
    		if (lr.label.equals("0") || lr.label.equals("key")) return keyType.getComponentType(path.childReference);
    		if (lr.label.equals("1") || lr.label.equals("value")) return valueType.getComponentType(path.childReference);
    	}
    	if (path instanceof NameReference) {
    		NameReference nr = (NameReference) path;
    		if (nr.name.equals("key")) return keyType.getComponentType(path.childReference);
    		if (nr.name.equals("value")) return valueType.getComponentType(path.childReference);
    	}
    	throw new IllegalArgumentException();
    }	
	@Override
	public void accept(Visitor1 v, Object obj) {
	    v.visit(this, obj);        
	}

	@Override
	public <T> T accept(Visitor<T> v) {
	    return v.visit(this);
	}

	@Override
	protected boolean deepEquals(Object obj,
			Set<IdentityPair<Datatype, Datatype>> compareHistory) {		
		if (this==obj) return true;
		if ( !hasEqualMetadata(obj) ) return false;
		if (obj instanceof MapType == false) return false;
		MapType other = (MapType) obj;
		
		return keyType.deepEquals(other.keyType, compareHistory) && 
			valueType.deepEquals(other.valueType, compareHistory);
	}
	
	@Override
	protected void collectSubtypes(Set<Datatype> subtypes, Set<Datatype> recursiveSubtypes) {
		keyType.collectSubtypes(subtypes, recursiveSubtypes);
		valueType.collectSubtypes(subtypes, recursiveSubtypes);
	}
	
	@Override
	public int hashCode() {
		if (keyType==this) return 0;
		if (valueType==this) return 0;
		return 0x232ae +
			ObjectUtils.hashCode(keyType) * 13 +
			ObjectUtils.hashCode(valueType) * 17;
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T extends Datatype> T getChildType(ChildReference reference) throws ReferenceException {
		if (reference==null) return (T) this;
		
		if (reference instanceof LabelReference) {
//			LabelReference lr = (LabelReference) reference;
			return keyType.getChildType(reference.getChildReference());
		} else if (reference instanceof KeyReference) {
//			KeyReference ref = (KeyReference) reference;
			return keyType.getChildType(reference.getChildReference());
		} 
		throw new ReferenceException(reference.getClass().getName()+" is not a reference of a map");	
	}
	
}

