package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.List;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer.NonRecursiveSerializer;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.util.Range;

public class FloatArraySerializer extends NonRecursiveSerializer {

	Range length;
	Integer fixedLength, fixedSize;
	
	public FloatArraySerializer(ArrayBinding binding)
	{			
		ArrayType arrayType = (ArrayType) binding.type();
		this.length = arrayType.getLength();
		
		if (length!=null && length.getLower().equals(length.getUpper()) && length.getLower().getValue()!=null)
		{
			fixedLength = length.getLower().getValue().intValue();
			fixedSize = fixedLength * 4;
		}
	}
	
	@Override
	public Object deserialize(DataInput in)
	throws IOException {
		int length = fixedLength != null ? fixedLength : in.readInt();
		if (length<0) throw new SerializationException("Cannot use negative array length");
		assertRemainingBytes(in, length*4L);					
		
		float[] array = new float[length];
		for(int i=0;i<array.length;++i)
			array[i] = in.readFloat();
		return array;
	}
	
	public Object deserializeToTry(DataInput in, List<Object> identities, Object obj) throws IOException
	{
		int length = fixedLength != null ? fixedLength : in.readInt();
		float[] array = (float[]) obj;
		if (length!=array.length) array = new float[ length ];
		assertRemainingBytes(in, length*4L);					
		
		for (int i=0; i<array.length;i++)
			array[i] = in.readFloat();
		
		return array;
	}
	
	@Override
	public void deserializeTo(DataInput in, Object obj) throws IOException {
		int length = fixedLength != null ? fixedLength : in.readInt();
		float[] array = (float[]) obj;
		if (length!=array.length) throw new SerializationException("primitive array is size immutable");
		assertRemainingBytes(in, length*4L);					
		for (int i=0; i<array.length;i++)
			array[i] = in.readFloat();
	}

	@Override
	public void skip(DataInput in)
	throws IOException {
		int length = fixedSize != null ? fixedLength : in.readInt();			
		in.skipBytes( length * 4 );
	}
	
	@Override
	public void serialize(DataOutput out, Object obj)
		throws IOException {
		float[] array = (float[])obj;
		if (fixedSize==null) 
			out.writeInt(array.length);
		for(float f : array)
			out.writeFloat(f);
	}

	@Override
	public Integer getConstantSize() {
		return fixedSize;
	}

	@Override
	public int getSize(Object obj) {
		if (fixedSize!=null) return fixedSize;
		float[] array = (float[])obj;			
		return 4 + 4 * array.length;
	}
	
	/**
	 * Get the number of bytes it takes to output length field of value <code>length</code>.
	 *  
	 * @param length length value
	 * @return bytes required
	 */
	public static int getSizeOfPutLength(int length)
	{
		if(length < 0x80) return 1;		
		if(length < 0x4080) return 2;
		if(length < 0x204000) return 3;
		if(length < 0x10200000) return 4;
		return 5;
	}
	
	@Override
	public int getMinSize() {
		return fixedSize != null ? fixedSize : 4;
	}
		
}