package org.simantics.databoard.primitives;


/**
 * Unsigned 16-bit integer value. The value is between 0 and 65535.
 *
 * @author Toni Kalajainen <toni.kalajainen@iki.fi>
 */
public abstract class UnsignedShort extends Number implements Comparable<Number> {

	private static final long serialVersionUID = 1L;

	public static final UnsignedShort MIN_VALUE, MAX_VALUE, ZERO;
	
	static long MASK = 0xFFFFl;
	public static final long L_MAX_VALUE = 0xFFFFL;
	public static final long L_MIN_VALUE = 0;

	/** Value container */
	int value;

    public static UnsignedShort valueOf(long value) {
		if (value>=0 && value<CACHE.length) return CACHE[(int)value];
		return new UnsignedShort.Immutable(value);    	
    }
		
    public static UnsignedShort fromBits(int intBits) {
		if (intBits>=0 && intBits<CACHE.length)	return CACHE[intBits];
    	UnsignedShort result = new UnsignedShort.Immutable();
    	result.value = intBits & 0xFFFF;
    	return result;
    }		

	public static class Mutable extends UnsignedShort {

		private static final long serialVersionUID = 1L;

		Mutable() {}
		
	    public Mutable(int value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = value;
	    }

	    public Mutable(long value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }

	    public Mutable(String stringValue) throws IllegalArgumentException {
	        long value = Long.parseLong(stringValue);
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }	
	    	    
	    public static UnsignedShort fromBits(int intBits) {
	    	UnsignedShort result = new Mutable();
	    	result.value = intBits & 0xFFFF;
	    	return result;
	    }		    
	    
	    public void setBits(int intBits) {
	    	this.value = intBits;
	    }
	    
	    public void setValue(int value) {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	    	this.value = value;
	    }
	    
	    public void setValue(long value) {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }	    
		
	}

	public static class Immutable extends UnsignedShort {

		private static final long serialVersionUID = 1L;
		
		Immutable() {}
		
	    public Immutable(int value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = value;
	    }

	    public Immutable(long value) throws IllegalArgumentException {
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }

	    public Immutable(String stringValue) throws IllegalArgumentException {
	        long value = Long.parseLong(stringValue);
	    	if ( value<L_MIN_VALUE || value>L_MAX_VALUE ) throw new IllegalArgumentException("Argument is not within range");
	        this.value = (int) value;
	    }	
	    	    
	    public static UnsignedShort fromBits(int intBits) {
	    	UnsignedShort result = new Immutable();
	    	result.value = intBits & 0xFFFF;
	    	return result;
	    }		    
		
	}
    
	
    public int toBits() {
    	return value;
    }
	
	@Override
	public int intValue() {
		return value;
	}
	
	@Override
	public long longValue() {
		return value & MASK;
	}
	
	@Override
	public float floatValue() {
		return value & MASK;
	}
	@Override
	public double doubleValue() {
		return value & MASK;
	}
	
    @Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (obj == this) return true;
		
		if (obj instanceof UnsignedShort == false) return false;
		UnsignedShort other = (UnsignedShort) obj;
		return value == other.value;
	}
    
    @Override
    public String toString() {
        return Long.toString(value & MASK);
    }
    
	@Override
	public int compareTo(Number obj) {
        return Long.signum( (value & MASK) - obj.longValue() );
	}
	
	@Override
	public int hashCode() {
		return value;
	}

	// Initialize Cache
	private static int CACHE_SIZE = 16;
	private static UnsignedShort.Immutable[] CACHE;
	static {
		CACHE = new UnsignedShort.Immutable[CACHE_SIZE];
		for (int i=0; i<CACHE_SIZE; i++) CACHE[i] = new UnsignedShort.Immutable(i);
		ZERO = MIN_VALUE = CACHE[0];
		MAX_VALUE = new UnsignedShort.Immutable(L_MAX_VALUE);
	}

}
