/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.method;

import java.util.concurrent.TimeUnit;

import org.simantics.databoard.Datatypes;

/**
 * MethodInterface is an interface for invoking methods.<p> 
 *
 * If MethodInterface is executed on TCP/IP socket, the current
 * Connection can be acquired with Connection#getCurrentConnection().
 * You can attach close listener this way.<p>
 * 
 * MethodInterface may be used from multiple-threads simultaneously.
 * 
 * @see Server puts a MethodInterface in a server socket 
 * @see Client creates a MethodInterface from a socket connection
 * @see Datatypes to create MethodInterface from an interface  
 * @see Datatypes to create MethodInterface implementation from an object
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 * @author Hannu Niemisto
 */
public interface MethodInterface {
	
	/**
	 * Get method descriptions
	 * 
	 * @return method descriptions
	 */
	Interface getInterface();

	/**
	 * Get an access to the implementation of a method. The binding is suggested by the producer. 
	 * 
	 * @param description method description
	 * @return method access
	 * @throws MethodNotSupportedException
	 */
	Method getMethod(MethodTypeDefinition description)
	throws MethodNotSupportedException;
	
	/**
	 * Get an access to the implementation of a method. The binding is suggested by the consumer.
	 * 
	 * @param binding binding to use
	 * @return method access
	 * @throws MethodNotSupportedException
	 */
	Method getMethod(MethodTypeBinding binding)
	throws MethodNotSupportedException;
	
	/**
	 * Access to the implementation of a method.
	 */
	public interface Method {
		AsyncResult invoke(Object request);
		MethodTypeBinding getMethodBinding();
	}
	
	public interface AsyncResult {
		
		/**
		 * Add a listener. It will be notified immediately if the response is
		 * available. 
		 *  
		 * @param listener (listener may not block) or null to remove listener
		 */
		void setListener(InvokeListener listener);
		
		/**
		 * 
		 * @return response or null
		 */
		Object getResponse();
		
		Object getExecutionError(); 
		
		InvokeException getInvokeException(); 
		
		AsyncRequestStatus getStatus(); 
		
		/**
		 * 
		 * @return response
		 * @throws InterruptedException block was canceled
		 * @throws InvokeException network error, e.g. IOException of MethodNotSupportedException
		 * @throws ExecutionError error that occured while executing the method 
		 */
		Object waitForResponse() 
		throws InvokeException, ExecutionError, InterruptedException;
		
		/**
		 * 
		 * @return response
		 * @throws InterruptedException 
		 * @throws InvokeException network error, e.g. IOException of MethodNotSupportedException
		 * @throws ExecutionError error that occured while executing the method 
		 */
		Object waitForResponse(long timeout, TimeUnit unit)
		throws InvokeException, ExecutionError, InterruptedException;
		
	}
	
	enum AsyncRequestStatus {Waiting, Succeed, Failed}
	
	public interface InvokeListener {
		void onCompleted(Object result);
		void onExecutionError(Object error);
		void onException(Exception cause);
	}
	
	/**
	 * A wrapper to an error that occured in the execution of the method.
	 */
	public static class ExecutionError extends Exception {
		private static final long serialVersionUID = 1L;
		Object error;
		public ExecutionError(Object error) {
			this.error = error;
		}
		public Object getError() {
			return error;
		}
	}
		
}

