/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.lang.reflect.Array;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Set;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.ByteBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.NumberType;
import org.simantics.databoard.util.IdentityPair;

/**
 * Binds Byte[] type to byte[] class.
 */
public final class ByteArrayBinding extends ArrayBinding {
	
	public static ByteArrayBinding createFrom(ArrayType type) {
		return new ByteArrayBinding(type, new ByteBindingDefault((ByteType) type.componentType));
	}

	public ByteArrayBinding(ArrayType type, Binding componentBinding) {
		super(type, componentBinding);
	}
	
	@Override
	public Object create() {
		return new byte[0];
	}
	
	@Override
	public Object create(int length, Iterator<Object> values) {
		byte[] array = new byte[length];
		for(int i=0;i<length;++i)
			array[i] = (Byte)values.next();
		return array;
	}

	public Object create(Object[] values) {
		byte[] result = new byte[values.length];
		for (int i=0; i<values.length; i++)
			result[i] = (Byte) values[i];
		return result;
	}	
	
	/**
	 * Create an array object.
	 * Note! The argument is consumed (is used in the result).
	 * 
	 * @param array
	 * @return an object that contains the array
	 */
	public Object create(byte[] array) {
		return array;
	}

	public Object create(Byte[] array) {
		int len = array.length;
		byte[] result = new byte[len];
		for (int i=0; i<len; i++)
			result[i] = array[i];
		return result;
	}
	
	@Override
    public Object createDefault()
    throws BindingException
    {
		NumberType nt = (NumberType) type().componentType;
		if (nt.getRange() == null) {
	    	return new byte[ type().minLength() ];
		}
		return super.createDefault();
    }	
	

	@Override
	public void readFrom(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		ByteBinding scb = (ByteBinding) sb.getComponentBinding();
		
		byte[] d = (byte[]) dst;
		if (d.length != sb.size(src)) throw new BindingException("byte[] is length immutable");
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
	}	
	
	@Override
	public Object readFromTry(Binding srcBinding, Object src, Object dst)
			throws BindingException {
		// Src Binding
		ArrayBinding sb = (ArrayBinding) srcBinding;
		// Src Component Binding
		ByteBinding scb = (ByteBinding) sb.getComponentBinding();
		
		byte[] d = (byte[]) dst;
		int srcSize = sb.size(src);
		if (d.length != srcSize) d = new byte[ srcSize ];
		
		for (int i=0; i<d.length; i++) {
			d[i] = scb.getValue_( sb.get(src, i) );
		}
		return d; 
	}		
	
	@Override
	public Object get(Object array, int index) throws BindingException 
	{
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", byte[] expected");
		return ((byte[])array)[index];
	}
	
	@Override
	public void getAll(Object array, Object[] result) throws BindingException {
		byte[] a = (byte[])array;
		for (int i=0; i<a.length; i++)
			result[i] = a[i];
	}

	@Override
	public void set(Object array, int index, Object value)
			throws BindingException {
		byte[] a = (byte[])array;
		a[index] = (Byte) value;
	}

	@Override
	public int size(Object array) 
	throws BindingException 
	{
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", byte[] expected");
		return ((byte[])array).length;
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof byte[];
	}
	
	@Override
	public boolean isImmutable() {
		return false;
	}	
	
	public byte[] getArray(Object array)
	throws BindingException
	{
		if (!isInstance(array)) throw new BindingException("Unexpected class "+array.getClass().getSimpleName()+", byte[] expected");
		return (byte[]) array;
	}

    @Override
    public int deepHashValue(Object value, IdentityHashMap<Object, Object> hashedObjects) throws BindingException {
    	int result = 1;
    	byte[] array = (byte[]) value;
    	for (int i=0; i<array.length; i++) {
    		result = 31*result + (int)array[i];
    	}
    	return result;
    }
    
    @Override
    public int deepCompare(Object o1, Object o2,
    		Set<IdentityPair<Object, Object>> compareHistory)
    		throws BindingException {
    	byte[] b1 = (byte[]) o1;
    	byte[] b2 = (byte[]) o2;
		// Compare Lengths
		int l1 = b1.length;
		int l2 = b2.length;
		int dif = l1 - l2;
		if (dif!=0) return dif;
		// Compare elements
		for (int i=0; i<l1; i++) {
			dif = b1[i] - b2[i];
			if (dif!=0) return dif;
		}
		return 0;
    }    

    @Override
    public void add(Object array, int index, Object element)
    		throws BindingException, IndexOutOfBoundsException {
    	throw new UnsupportedOperationException();
    }
    
	@Override
	public void remove(Object array, int index, int count) throws BindingException {
    	throw new UnsupportedOperationException();
	}
    
	@Override
	public void setSize(Object array, int newSize) throws BindingException {
		int oldSize = Array.getLength(array);
		if (oldSize==newSize) return;
		throw new BindingException("byte[] is length immutable");
	}

	@Override
	public boolean isResizable() {
		return false;
	}
	
}
